package ru.yandex.solomon.acl.db.memory;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.model.GroupMember;
import ru.yandex.solomon.core.exceptions.ConflictException;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class InMemoryGroupMemberDao implements GroupMemberDao {
    private final ConcurrentMap<String, List<GroupMember>> groupToMembers = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return runAsync(() -> {
        });
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return runAsync(groupToMembers::clear);
    }

    @Override
    public CompletableFuture<Void> create(Collection<GroupMember> groupMembers) {
        return runAsync(() -> {
            groupMembers.forEach(
                    groupMember -> groupToMembers.computeIfAbsent(groupMember.groupId(), s -> new CopyOnWriteArrayList<>()).add(groupMember));
        });
    }

    @Override
    public CompletableFuture<Void> delete(Collection<GroupMember> groupMembers) {
        return supplyAsync(() -> {
            for (var groupMember : groupMembers) {
                var members = groupToMembers.getOrDefault(groupMember.groupId(), Collections.emptyList());
                members.remove(groupMember);
                if (members.isEmpty()) {
                    groupToMembers.remove(groupMember.groupId(), members);
                }
            }
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> deleteByGroupId(String groupId) {
        return supplyAsync(() -> {
            if (!groupToMembers.remove(groupId, groupToMembers.get(groupId))) {
                throw new ConflictException("concurrent modification");
            }
            return null;
        });
    }

    @Override
    public CompletableFuture<List<GroupMember>> find(String groupId) {
        return supplyAsync(() -> groupToMembers.getOrDefault(groupId, Collections.emptyList()));
    }

    @Override
    public CompletableFuture<List<GroupMember>> getAll() {
        return CompletableFuture.completedFuture(groupToMembers.values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<Boolean> exist(String groupId) {
        return supplyAsync(() -> groupToMembers.containsKey(groupId));
    }

}
