package ru.yandex.solomon.acl.db.memory;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class InMemoryProjectAclEntryDao implements ProjectAclEntryDao {
    private final ConcurrentMap<ProjectAclEntry.Id, ProjectAclEntry> aclById = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return runAsync(() -> {
        });
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return runAsync(aclById::clear);
    }

    @Override
    public CompletableFuture<Optional<ProjectAclEntry>> find(String projectId, String uid, AclUidType type) {
        var id = ProjectAclEntry.compositeId(projectId, uid, type);
        return supplyAsync(() -> Optional.ofNullable(aclById.get(id)).map(ProjectAclEntry::copy));
    }

    @Override
    public CompletableFuture<List<ProjectAclEntry>> getAll() {
        return CompletableFuture.supplyAsync(() -> List.copyOf(aclById.values()));
    }

    @Override
    public CompletableFuture<TokenBasePage<ProjectAclEntry>> list(String projectId, int pageSize, String pageToken) {
        return CompletableFuture.supplyAsync(() -> {
            var list = aclById.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .filter(idProjectAclEntryEntry -> idProjectAclEntryEntry.getKey().projectId().equals(projectId))
                    .map(Map.Entry::getValue)
                    .collect(Collectors.toList());
            return toPageResult(list, pageSize, pageToken);
        });
    }

    @Override
    public CompletableFuture<Boolean> create(ProjectAclEntry projectAclEntry) {
        return supplyAsync(() -> aclById.putIfAbsent(projectAclEntry.getCompositeId(), projectAclEntry) == null);
    }

    @Override
    public CompletableFuture<Void> update(ProjectAclEntry projectAclEntry) {
        return runAsync(() -> {
            ProjectAclEntry previous = aclById.get(projectAclEntry.getCompositeId());
            if (previous.getVersion() != projectAclEntry.getVersion()) {
                throw outOfDate(projectAclEntry);
            }
            var result = new ProjectAclEntry
                    (
                            projectAclEntry.getProjectId(),
                            projectAclEntry.getUid(),
                            projectAclEntry.getType(),
                            projectAclEntry.getRoles(),
                            projectAclEntry.getVersion() + 1
                    );
            boolean replace = aclById.replace(projectAclEntry.getCompositeId(), previous, result);
            if (!replace) {
                throw outOfDate(projectAclEntry);
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> delete(ProjectAclEntry projectAclEntry) {
        return supplyAsync(() -> {
            ProjectAclEntry previous = aclById.get(projectAclEntry.getCompositeId());
            if (previous != null && previous.getVersion() != projectAclEntry.getVersion()) {
                return false;
            }
            return aclById.remove(projectAclEntry.getCompositeId(), previous);
        });
    }

    private static ConflictException outOfDate(ProjectAclEntry entry) {
        String message = String.format(
                "ProjectAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }

    private TokenBasePage<ProjectAclEntry> toPageResult(List<ProjectAclEntry> matched, int pageSize, String pageToken) {
        if (pageSize <= 0) {
            pageSize = 100;
        } else {
            pageSize = Math.min(pageSize, 1000);
        }

        int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);
        int nextOffset = Math.min(offset + pageSize, matched.size());
        var list = matched.subList(offset, nextOffset);
        if (nextOffset >= matched.size()) {
            return new TokenBasePage<>(list, "");
        }
        return new TokenBasePage<>(list, Integer.toString(nextOffset));
    }
}
