package ru.yandex.solomon.acl.db.memory;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class InMemoryServiceProviderAclEntryDao implements ServiceProviderAclEntryDao {
    private final ConcurrentMap<ServiceProviderAclEntry.Id, ServiceProviderAclEntry> aclById = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return runAsync(() -> {
        });
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return runAsync(aclById::clear);
    }

    @Override
    public CompletableFuture<Optional<ServiceProviderAclEntry>> find(String serviceProviderId, String uid, AclUidType type) {
        var id = ServiceProviderAclEntry.compositeId(serviceProviderId, uid, type);
        return supplyAsync(() -> Optional.ofNullable(aclById.get(id)).map(ServiceProviderAclEntry::copy));
    }

    @Override
    public CompletableFuture<List<ServiceProviderAclEntry>> getAll() {
        return CompletableFuture.supplyAsync(() -> List.copyOf(aclById.values()));
    }

    @Override
    public CompletableFuture<TokenBasePage<ServiceProviderAclEntry>> list(String serviceProviderId, int pageSize, String pageToken) {
        return CompletableFuture.supplyAsync(() -> {
            var list = aclById.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .filter(idServiceProviderAclEntryEntry -> idServiceProviderAclEntryEntry.getKey().serviceProviderId().equals(serviceProviderId))
                    .map(Map.Entry::getValue)
                    .collect(Collectors.toList());
            return toPageResult(list, pageSize, pageToken);
        });
    }

    @Override
    public CompletableFuture<Boolean> create(ServiceProviderAclEntry ServiceProviderAclEntry) {
        return supplyAsync(() -> aclById.putIfAbsent(ServiceProviderAclEntry.getCompositeId(), ServiceProviderAclEntry) == null);
    }

    @Override
    public CompletableFuture<Void> update(ServiceProviderAclEntry ServiceProviderAclEntry) {
        return runAsync(() -> {
            ServiceProviderAclEntry previous = aclById.get(ServiceProviderAclEntry.getCompositeId());
            if (previous.getVersion() != ServiceProviderAclEntry.getVersion()) {
                throw outOfDate(ServiceProviderAclEntry);
            }
            var result = new ServiceProviderAclEntry
                    (
                            ServiceProviderAclEntry.getServiceProviderId(),
                            ServiceProviderAclEntry.getUid(),
                            ServiceProviderAclEntry.getType(),
                            ServiceProviderAclEntry.getRoles(),
                            ServiceProviderAclEntry.getVersion() + 1
                    );
            boolean replace = aclById.replace(ServiceProviderAclEntry.getCompositeId(), previous, result);
            if (!replace) {
                throw outOfDate(ServiceProviderAclEntry);
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> delete(ServiceProviderAclEntry ServiceProviderAclEntry) {
        return supplyAsync(() -> {
            ServiceProviderAclEntry previous = aclById.get(ServiceProviderAclEntry.getCompositeId());
            if (previous != null && previous.getVersion() != ServiceProviderAclEntry.getVersion()) {
                return false;
            }
            return aclById.remove(ServiceProviderAclEntry.getCompositeId(), previous);
        });
    }

    private static ConflictException outOfDate(ServiceProviderAclEntry entry) {
        String message = String.format(
                "ServiceProviderAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }

    private TokenBasePage<ServiceProviderAclEntry> toPageResult(List<ServiceProviderAclEntry> matched, int pageSize, String pageToken) {
        if (pageSize <= 0) {
            pageSize = 100;
        } else {
            pageSize = Math.min(pageSize, 1000);
        }

        int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);
        int nextOffset = Math.min(offset + pageSize, matched.size());
        var list = matched.subList(offset, nextOffset);
        if (nextOffset >= matched.size()) {
            return new TokenBasePage<>(list, "");
        }
        return new TokenBasePage<>(list, Integer.toString(nextOffset));
    }
}
