package ru.yandex.solomon.acl.db.memory;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class InMemorySystemAclEntryDao implements SystemAclEntryDao {
    private final ConcurrentMap<SystemAclEntry.Id, SystemAclEntry> aclById = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return runAsync(() -> {
        });
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return runAsync(aclById::clear);
    }

    @Override
    public CompletableFuture<Optional<SystemAclEntry>> find(String uid, AclUidType type) {
        var id = SystemAclEntry.compositeId(uid, type);
        return supplyAsync(() -> Optional.ofNullable(aclById.get(id)).map(SystemAclEntry::copy));
    }

    @Override
    public CompletableFuture<Boolean> create(SystemAclEntry systemAclEntry) {
        return supplyAsync(() -> aclById.putIfAbsent(systemAclEntry.getCompositeId(), systemAclEntry) == null);
    }

    @Override
    public CompletableFuture<List<SystemAclEntry>> getAll() {
        return CompletableFuture.supplyAsync(() -> List.copyOf(aclById.values()));
    }

    @Override
    public CompletableFuture<Void> update(SystemAclEntry systemAclEntry) {
        return runAsync(() -> {
            SystemAclEntry previous = aclById.get(systemAclEntry.getCompositeId());
            if (previous.getVersion() != systemAclEntry.getVersion()) {
                throw outOfDate(systemAclEntry);
            }
            var result = new SystemAclEntry
                    (
                            systemAclEntry.getUid(),
                            systemAclEntry.getType(),
                            systemAclEntry.getRoles(),
                            systemAclEntry.getVersion() + 1
                    );
            boolean replace = aclById.replace(systemAclEntry.getCompositeId(), previous, result);
            if (!replace) {
                throw outOfDate(systemAclEntry);
            }
        });

    }

    @Override
    public CompletableFuture<Boolean> delete(SystemAclEntry systemAclEntry) {
        return supplyAsync(() -> {
            SystemAclEntry previous = aclById.get(systemAclEntry.getCompositeId());
            if (previous != null && previous.getVersion() != systemAclEntry.getVersion()) {
                return false;
            }
            return aclById.remove(systemAclEntry.getCompositeId(), previous);
        });
    }

    private static ConflictException outOfDate(SystemAclEntry entry) {
        String message = String.format(
                "SystemAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }

}
