package ru.yandex.solomon.acl.db.ydb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;

import ru.yandex.bolts.function.Function;
import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.model.GroupMember;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveType.utf8;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.acl.db.ydb.queries.GroupMemberQueries.DELETE_BY_GROUP_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.GroupMemberQueries.DELETE_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.GroupMemberQueries.EXIST_BY_GROUP_ID_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.GroupMemberQueries.FIND_BY_GROUP_ID_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.GroupMemberQueries.UPSERT_TEMPLATE;

/**
 * @author Alexey Trushkin
 */
@SuppressWarnings("rawtypes")
@ParametersAreNonnullByDefault
public class YdbGroupMemberDao implements GroupMemberDao {

    private static final String GROUP_MEMBER_TABLE = "GroupMember";
    static final int BATCH_SIZE = 1000;
    private static final StructType TYPE = StructType.of(
            "groupId", PrimitiveType.utf8(),
            "userId", PrimitiveType.utf8());
    private static final ListType LIST_TYPE = ListType.of(TYPE);

    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;

    private final String deleteQuery;
    private final String upsertQuery;
    private final String findByGroupIdQuery;
    private final String existByGroupIdQuery;
    private final String deleteByGroupQuery;

    public YdbGroupMemberDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.root = root;
        this.schemeClient = schemeClient;
        this.tablePath = root + GROUP_MEMBER_TABLE;
        this.table = new Table(tableClient, tablePath);

        this.upsertQuery = String.format(UPSERT_TEMPLATE, LIST_TYPE, tablePath);
        this.findByGroupIdQuery = String.format(FIND_BY_GROUP_ID_TEMPLATE, tablePath);
        this.existByGroupIdQuery = String.format(EXIST_BY_GROUP_ID_TEMPLATE, tablePath);
        this.deleteQuery = String.format(DELETE_TEMPLATE, LIST_TYPE, tablePath);
        this.deleteByGroupQuery = String.format(DELETE_BY_GROUP_TEMPLATE, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<List<GroupMember>> find(String groupId) {
        try {
            Params params = Params.of("$groupId", utf8(groupId));
            return table.queryList(findByGroupIdQuery, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<GroupMember>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exist(String groupId) {
        try {
            Params params = Params.of("$groupId", utf8(groupId));
            return table.queryBool(existByGroupIdQuery, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByGroupId(String groupId) {
        try {
            Params params = Params.of("$groupId", utf8(groupId));
            return table.queryVoid(deleteByGroupQuery, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> create(Collection<GroupMember> groupMembers) {
        return batchOperation(groupMembers, this::insertBatch);
    }

    @Override
    public CompletableFuture<Void> delete(Collection<GroupMember> groupMembers) {
        return batchOperation(groupMembers, this::deleteBatch);
    }

    private CompletableFuture<Void> insertBatch(List<Value> values) {
        Params params = Params.of("$rows", LIST_TYPE.newValue(values));
        return table.execute(upsertQuery, params)
                .thenAccept(result -> result.expect("cannot insert group members"));
    }

    private CompletableFuture<Void> deleteBatch(List<Value> values) {
        Params params = Params.of("$rows", LIST_TYPE.newValue(values));
        return table.execute(deleteQuery, params)
                .thenAccept(result -> result.expect("cannot delete group members"));
    }

    private CompletableFuture<Void> batchOperation(
            Collection<GroupMember> groupMembers,
            Function<List<Value>, CompletableFuture<Void>> operation)
    {
        if (groupMembers.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        List<Value> values = new ArrayList<>(groupMembers.size());
        for (var groupMember : groupMembers) {
            values.add(TYPE.newValue(
                    "groupId", utf8(groupMember.groupId()),
                    "userId", utf8(groupMember.userId())));
        }
        if (values.size() <= BATCH_SIZE) {
            return operation.apply(values);
        }
        // split all values into N batches and insert them with inFlight=1
        CompletableFuture<Void> future = CompletableFuture.completedFuture(null);
        for (var batch : Lists.partition(values, BATCH_SIZE)) {
            future = future.thenCompose(aVoid -> operation.apply(batch));
        }
        return future;
    }

    private static final class Table extends YdbTable<String, GroupMember> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("groupId", utf8())
                    .addNullableColumn("userId", utf8())
                    .setPrimaryKeys("groupId", "userId")
                    .build();
        }

        @Override
        protected String getId(GroupMember groupMember) {
            return groupMember.getCompositeId();
        }

        @Override
        protected Params toParams(GroupMember groupMember) {
            return Params.create()
                    .put("$groupId", utf8(groupMember.groupId()))
                    .put("$userId", utf8(groupMember.userId()));
        }

        @Override
        protected GroupMember mapFull(ResultSetReader r) {
            return new GroupMember(
                    r.getColumn("groupId").getUtf8(),
                    r.getColumn("userId").getUtf8());
        }

        @Override
        protected GroupMember mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
