package ru.yandex.solomon.acl.db.ydb;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;

import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveType.uint32;
import static com.yandex.ydb.table.values.PrimitiveType.utf8;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.acl.db.ydb.queries.ProjectAclEntryQueries.DELETE_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ProjectAclEntryQueries.FIND_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ProjectAclEntryQueries.INSERT_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ProjectAclEntryQueries.LIST_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ProjectAclEntryQueries.UPDATE_TEMPLATE;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.setFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class YdbProjectAclEntryDao implements ProjectAclEntryDao {

    private static final String PROJECT_ACL_ENTRY_TABLE = "ProjectAclEntry";
    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;

    private final String deleteQuery;
    private final String updateQuery;
    private final String insertQuery;
    private final String findQuery;
    private final String listQuery;

    public YdbProjectAclEntryDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.root = root;
        this.schemeClient = schemeClient;
        this.tablePath = root + PROJECT_ACL_ENTRY_TABLE;
        this.table = new Table(tableClient, tablePath);

        this.insertQuery = String.format(INSERT_TEMPLATE, tablePath);
        this.updateQuery = String.format(UPDATE_TEMPLATE, tablePath, tablePath);
        this.findQuery = String.format(FIND_TEMPLATE, tablePath);
        this.deleteQuery = String.format(DELETE_TEMPLATE, tablePath, tablePath);
        this.listQuery = String.format(LIST_TEMPLATE, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Boolean> create(ProjectAclEntry projectAclEntry) {
        try {
            return table.insertOne(insertQuery, projectAclEntry);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> update(ProjectAclEntry projectAclEntry) {
        try {
            Params params = Params.of(
                    "$projectId", utf8(projectAclEntry.getProjectId()),
                    "$uid", utf8(projectAclEntry.getUid()),
                    "$type", utf8(projectAclEntry.getType().name()),
                    "$roles", utf8(toTsv(projectAclEntry.getRoles())),
                    "$version", uint32(projectAclEntry.getVersion())
            );
            return table.queryBool(updateQuery, params)
                    .thenCompose(updated -> updated ?
                            CompletableFuture.completedFuture(null) :
                            CompletableFuture.failedFuture(outOfDate(projectAclEntry)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(ProjectAclEntry projectAclEntry) {
        try {
            Params params = Params.of(
                    "$projectId", utf8(projectAclEntry.getProjectId()),
                    "$uid", utf8(projectAclEntry.getUid()),
                    "$type", utf8(projectAclEntry.getType().name()),
                    "$version", uint32(projectAclEntry.getVersion())
            );
            return table.queryBool(deleteQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<ProjectAclEntry>> find(String projectId, String uid, AclUidType type) {
        try {
            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$uid", utf8(uid),
                    "$type", utf8(type.name())
            );
            return table.queryOne(findQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ProjectAclEntry>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<ProjectAclEntry>> list(String projectId, int pageSize, String pageToken) {
        try {
            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(listQuery, params)
                    .thenApply(result -> {
                        final String nextPageToken;
                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }
                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private static final class Table extends YdbTable<ProjectAclEntry.Id, ProjectAclEntry> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("projectId", utf8())
                    .addNullableColumn("uid", utf8())
                    .addNullableColumn("type", utf8())
                    .addNullableColumn("roles", utf8())
                    .addNullableColumn("version", uint32())
                    .setPrimaryKeys("projectId", "uid", "type")
                    .build();
        }

        @Override
        protected ProjectAclEntry.Id getId(ProjectAclEntry projectAclEntry) {
            return projectAclEntry.getCompositeId();
        }

        @Override
        protected Params toParams(ProjectAclEntry projectAclEntry) {
            return Params.create()
                    .put("$projectId", utf8(projectAclEntry.getProjectId()))
                    .put("$uid", utf8(projectAclEntry.getUid()))
                    .put("$type", utf8(projectAclEntry.getType().name()))
                    .put("$roles", utf8(toTsv(projectAclEntry.getRoles())))
                    .put("$version", uint32(projectAclEntry.getVersion()));
        }

        @Override
        protected ProjectAclEntry mapFull(ResultSetReader r) {
            return new ProjectAclEntry(
                    r.getColumn("projectId").getUtf8(),
                    r.getColumn("uid").getUtf8(),
                    AclUidType.valueOf(r.getColumn("type").getUtf8()),
                    setFromTsv(r.getColumn("roles").getUtf8()),
                    (int) r.getColumn("version").getUint32());
        }

        @Override
        protected ProjectAclEntry mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static ConflictException outOfDate(ProjectAclEntry entry) {
        String message = String.format(
                "ProjectAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }
}
