package ru.yandex.solomon.acl.db.ydb;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.acl.db.ydb.queries.ServiceProviderAclEntryQueries.DELETE_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ServiceProviderAclEntryQueries.FIND_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ServiceProviderAclEntryQueries.INSERT_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ServiceProviderAclEntryQueries.LIST_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.ServiceProviderAclEntryQueries.UPDATE_TEMPLATE;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.setFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class YdbServiceProviderAclEntryDao implements ServiceProviderAclEntryDao {

    private static final String TABLE_NAME = "ServiceProviderAclEntry";
    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;

    private final String deleteQuery;
    private final String updateQuery;
    private final String insertQuery;
    private final String findQuery;
    private final String listQuery;

    public YdbServiceProviderAclEntryDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.root = root;
        this.schemeClient = schemeClient;
        this.tablePath = root + TABLE_NAME;
        this.table = new Table(tableClient, tablePath);

        this.insertQuery = String.format(INSERT_TEMPLATE, tablePath);
        this.updateQuery = String.format(UPDATE_TEMPLATE, tablePath, tablePath);
        this.findQuery = String.format(FIND_TEMPLATE, tablePath);
        this.deleteQuery = String.format(DELETE_TEMPLATE, tablePath, tablePath);
        this.listQuery = String.format(LIST_TEMPLATE, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Boolean> create(ServiceProviderAclEntry entry) {
        try {
            return table.insertOne(insertQuery, entry);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> update(ServiceProviderAclEntry entry) {
        try {
            Params params = Params.of(
                    "$serviceProviderId", utf8(entry.getServiceProviderId()),
                    "$uid", utf8(entry.getUid()),
                    "$type", utf8(entry.getType().name()),
                    "$roles", utf8(toTsv(entry.getRoles())),
                    "$version", uint32(entry.getVersion())
            );
            return table.queryBool(updateQuery, params)
                    .thenCompose(updated -> updated ?
                            CompletableFuture.completedFuture(null) :
                            CompletableFuture.failedFuture(outOfDate(entry)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(ServiceProviderAclEntry entry) {
        try {
            Params params = Params.of(
                    "$serviceProviderId", utf8(entry.getServiceProviderId()),
                    "$uid", utf8(entry.getUid()),
                    "$type", utf8(entry.getType().name()),
                    "$version", uint32(entry.getVersion())
            );
            return table.queryBool(deleteQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<ServiceProviderAclEntry>> find(String serviceProviderId, String uid, AclUidType type) {
        try {
            Params params = Params.of(
                    "$serviceProviderId", utf8(serviceProviderId),
                    "$uid", utf8(uid),
                    "$type", utf8(type.name())
            );
            return table.queryOne(findQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ServiceProviderAclEntry>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<ServiceProviderAclEntry>> list(String serviceProviderId, int pageSize, String pageToken) {
        try {
            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$serviceProviderId", utf8(serviceProviderId),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(listQuery, params)
                    .thenApply(result -> {
                        final String nextPageToken;
                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }
                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private static final class Table extends YdbTable<ServiceProviderAclEntry.Id, ServiceProviderAclEntry> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("serviceProviderId", PrimitiveType.utf8())
                    .addNullableColumn("uid", PrimitiveType.utf8())
                    .addNullableColumn("type", PrimitiveType.utf8())
                    .addNullableColumn("roles", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.uint32())
                    .setPrimaryKeys("serviceProviderId", "uid", "type")
                    .build();
        }

        @Override
        protected ServiceProviderAclEntry.Id getId(ServiceProviderAclEntry entry) {
            return entry.getCompositeId();
        }

        @Override
        protected Params toParams(ServiceProviderAclEntry entry) {
            return Params.create()
                    .put("$serviceProviderId", utf8(entry.getServiceProviderId()))
                    .put("$uid", utf8(entry.getUid()))
                    .put("$type", utf8(entry.getType().name()))
                    .put("$roles", utf8(toTsv(entry.getRoles())))
                    .put("$version", uint32(entry.getVersion()));
        }

        @Override
        protected ServiceProviderAclEntry mapFull(ResultSetReader r) {
            return new ServiceProviderAclEntry(
                    r.getColumn("serviceProviderId").getUtf8(),
                    r.getColumn("uid").getUtf8(),
                    AclUidType.valueOf(r.getColumn("type").getUtf8()),
                    setFromTsv(r.getColumn("roles").getUtf8()),
                    (int) r.getColumn("version").getUint32());
        }

        @Override
        protected ServiceProviderAclEntry mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static ConflictException outOfDate(ServiceProviderAclEntry entry) {
        String message = String.format(
                "ServiceProviderAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }
}
