package ru.yandex.solomon.acl.db.ydb;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;

import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveType.uint32;
import static com.yandex.ydb.table.values.PrimitiveType.utf8;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.acl.db.ydb.queries.SystemAclEntryQueries.DELETE_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.SystemAclEntryQueries.FIND_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.SystemAclEntryQueries.INSERT_TEMPLATE;
import static ru.yandex.solomon.acl.db.ydb.queries.SystemAclEntryQueries.UPDATE_TEMPLATE;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.setFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class YdbSystemAclEntryDao implements SystemAclEntryDao {

    private static final String SYSTEM_ACL_ENTRY_TABLE = "SystemAclEntry";
    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;

    private final String deleteQuery;
    private final String updateQuery;
    private final String findQuery;
    private final String insertQuery;

    public YdbSystemAclEntryDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.root = root;
        this.schemeClient = schemeClient;
        this.tablePath = root + SYSTEM_ACL_ENTRY_TABLE;
        this.table = new Table(tableClient, tablePath);

        this.insertQuery = String.format(INSERT_TEMPLATE, tablePath);
        this.updateQuery = String.format(UPDATE_TEMPLATE, tablePath, tablePath);
        this.findQuery = String.format(FIND_TEMPLATE, tablePath);
        this.deleteQuery = String.format(DELETE_TEMPLATE, tablePath, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Boolean> create(SystemAclEntry systemAclEntry) {
        try {
            return table.insertOne(insertQuery, systemAclEntry);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<SystemAclEntry>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> update(SystemAclEntry systemAclEntry) {
        try {
            Params params = Params.of(
                    "$uid", utf8(systemAclEntry.getUid()),
                    "$type", utf8(systemAclEntry.getType().name()),
                    "$roles", utf8(toTsv(systemAclEntry.getRoles())),
                    "$version", uint32(systemAclEntry.getVersion())
            );
            return table.queryBool(updateQuery, params)
                    .thenCompose(updated -> updated ?
                            CompletableFuture.completedFuture(null) :
                            CompletableFuture.failedFuture(outOfDate(systemAclEntry)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(SystemAclEntry systemAclEntry) {
        try {
            Params params = Params.of(
                    "$uid", utf8(systemAclEntry.getUid()),
                    "$type", utf8(systemAclEntry.getType().name()),
                    "$version", uint32(systemAclEntry.getVersion())
            );
            return table.queryBool(deleteQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<SystemAclEntry>> find(String uid, AclUidType type) {
        try {
            Params params = Params.of(
                    "$uid", utf8(uid),
                    "$type", utf8(type.name())
            );
            return table.queryOne(findQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private static final class Table extends YdbTable<SystemAclEntry.Id, SystemAclEntry> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("uid", utf8())
                    .addNullableColumn("type", utf8())
                    .addNullableColumn("roles", utf8())
                    .addNullableColumn("version", uint32())
                    .setPrimaryKeys("uid", "type")
                    .build();
        }

        @Override
        protected SystemAclEntry.Id getId(SystemAclEntry systemAclEntry) {
            return systemAclEntry.getCompositeId();
        }

        @Override
        protected Params toParams(SystemAclEntry systemAclEntry) {
            return Params.create()
                    .put("$uid", utf8(systemAclEntry.getUid()))
                    .put("$type", utf8(systemAclEntry.getType().name()))
                    .put("$roles", utf8(toTsv(systemAclEntry.getRoles())))
                    .put("$version", uint32(systemAclEntry.getVersion()));
        }

        @Override
        protected SystemAclEntry mapFull(ResultSetReader r) {
            return new SystemAclEntry(
                    r.getColumn("uid").getUtf8(),
                    AclUidType.valueOf(r.getColumn("type").getUtf8()),
                    setFromTsv(r.getColumn("roles").getUtf8()),
                    (int) r.getColumn("version").getUint32());
        }

        @Override
        protected SystemAclEntry mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static ConflictException outOfDate(SystemAclEntry entry) {
        String message = String.format(
                "SystemAclEntry (%s) with version %s is out of date",
                entry.getCompositeId(),
                entry.getVersion()
        );
        return new ConflictException(message);
    }
}
