package ru.yandex.solomon.acl.db;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.solomon.acl.db.model.GroupMember;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractGroupMemberDaoTest {

    private static final String USER_ID1 = "user 1";
    private static final String USER_ID2 = "user 2";
    private static final String USER_ID3 = "user 3";
    private static final String GROUP_ID1 = "group 1";
    private static final String GROUP_ID2 = "group 2";

    protected abstract GroupMemberDao getDao();

    @Test
    public void create() {
        createSync(groupMember(USER_ID1, GROUP_ID1));
        assertEquals(List.of(groupMember(USER_ID1, GROUP_ID1)), findByGroupIdSync(GROUP_ID1));
    }

    @Test
    public void createMultiple() {
        createSync(List.of(
                groupMember(USER_ID1, GROUP_ID1),
                groupMember(USER_ID2, GROUP_ID1),
                groupMember(USER_ID2, GROUP_ID2)
        ));
        List<GroupMember> group1 = findByGroupIdSync(GROUP_ID1);
        List<GroupMember> group2 = findByGroupIdSync(GROUP_ID2);
        assertEquals(2, group1.size());
        assertTrue(group1.contains(groupMember(USER_ID1, GROUP_ID1)));
        assertTrue(group1.contains(groupMember(USER_ID2, GROUP_ID1)));
        assertEquals(1, group2.size());
        assertTrue(group2.contains(groupMember(USER_ID2, GROUP_ID2)));
    }

    @Test
    public void findByGroupId() {
        assertEquals(List.of(), findByGroupIdSync(GROUP_ID1));

        createSync(groupMember(USER_ID1, GROUP_ID1));
        assertEquals(List.of(), findByGroupIdSync(GROUP_ID2));

        assertEquals(List.of(groupMember(USER_ID1, GROUP_ID1)), findByGroupIdSync(GROUP_ID1));
    }

    @Test
    public void findByGroupId_multipleGroups() {
        assertEquals(List.of(), findByGroupIdSync(GROUP_ID1));
        assertEquals(List.of(), findByGroupIdSync(GROUP_ID2));

        createSync(groupMember(USER_ID1, GROUP_ID1));
        createSync(groupMember(USER_ID2, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID2));

        assertEquals(findByGroupIdSync(GROUP_ID1).size(), 3);
        assertEquals(findByGroupIdSync(GROUP_ID2).size(), 1);
    }

    @Test
    public void delete() {
        createSync(groupMember(USER_ID1, GROUP_ID1));
        createSync(groupMember(USER_ID2, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID2));

        deleteSync(List.of());

        assertEquals(findByGroupIdSync(GROUP_ID2).size(), 1);
        deleteSync(List.of(groupMember(USER_ID3, GROUP_ID2)));
        assertEquals(List.of(), findByGroupIdSync(GROUP_ID2));

        assertEquals(findByGroupIdSync(GROUP_ID1).size(), 3);
        deleteSync(List.of(
                groupMember(USER_ID1, GROUP_ID1),
                groupMember(USER_ID3, GROUP_ID1)));

        assertEquals(List.of(groupMember(USER_ID2, GROUP_ID1)), findByGroupIdSync(GROUP_ID1));
    }

    @Test
    public void deleteByGroup() {
        createSync(groupMember(USER_ID1, GROUP_ID1));
        createSync(groupMember(USER_ID2, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID2));

        deleteSync(GROUP_ID1);

        assertEquals(findByGroupIdSync(GROUP_ID2).size(), 1);
        assertEquals(findByGroupIdSync(GROUP_ID1).size(), 0);
    }

    @Test
    public void getAll() {
        assertEquals(Map.of(), getAllSync());

        createSync(groupMember(USER_ID1, GROUP_ID1));
        createSync(groupMember(USER_ID2, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID1));
        createSync(groupMember(USER_ID3, GROUP_ID2));

        var actual = getAllSync();

        assertEquals(Map.of(
                GROUP_ID1, List.of(groupMember(USER_ID1, GROUP_ID1), groupMember(USER_ID2, GROUP_ID1), groupMember(USER_ID3, GROUP_ID1)),
                GROUP_ID2, List.of(groupMember(USER_ID3, GROUP_ID2))
        ), actual);
    }

    private void createSync(GroupMember groupMember) {
        join(getDao().create(List.of(groupMember)));
    }

    private void createSync(List<GroupMember> groupMembers) {
        join(getDao().create(groupMembers));
    }

    private void deleteSync(Collection<GroupMember> groupMembers) {
        join(getDao().delete(groupMembers));
    }

    private void deleteSync(String groupId) {
        join(getDao().deleteByGroupId(groupId));
    }

    private Map<String, List<GroupMember>> getAllSync() {
        return join(getDao().getAll()
                .thenApply(groupMembers ->
                        groupMembers.stream().collect(Collectors.groupingBy(GroupMember::groupId))));
    }

    private List<GroupMember> findByGroupIdSync(String groupId) {
        return join(getDao().find(groupId));
    }

    private GroupMember groupMember(String userId, String groupId) {
        return new GroupMember(groupId, userId);
    }
}
