package ru.yandex.solomon.acl.db;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractProjectAclEntryDaoTest {

    private static final String PROJECT_ID = "its a project";
    private static final String PROJECT_ID2 = "its another project";
    private static final String UID = "some uid";
    private static final Set<String> ROLES = Set.of("role 1", "role 2", "role 3");

    protected abstract ProjectAclEntryDao getDao();

    @Test
    public void create() {
        assertTrue(createSync(projectAclEntry()));
    }

    @Test
    public void update() {
        ProjectAclEntry projectAclEntry = projectAclEntry();
        assertTrue(createSync(projectAclEntry));

        projectAclEntry.setRoles(Set.of("role 4"));
        updateSync(projectAclEntry);
        var actual = findSync(PROJECT_ID, UID, AclUidType.USER);
        assertTrue(actual.isPresent());
        assertEquals(PROJECT_ID, actual.get().getProjectId());
        assertEquals(UID, actual.get().getUid());
        assertEquals(Set.of("role 4"), actual.get().getRoles());
        assertEquals(AclUidType.USER, actual.get().getType());
    }

    @Test(expected = CompletionException.class)
    public void concurrentActions() {
        assertTrue(createSync(projectAclEntry()));

        //now version 1
        updateSync(projectAclEntry());
        Optional<ProjectAclEntry> projectAclEntry = findSync(PROJECT_ID, UID, AclUidType.USER);
        //update to version 2
        updateSync(projectAclEntry.get());
        assertFalse(deleteSync(projectAclEntry.get()));

        //update with version 1, but actual version 2
        updateSync(projectAclEntry.get());
    }

    @Test
    public void find() {
        ProjectAclEntry projectAclEntry = projectAclEntry();
        assertTrue(createSync(projectAclEntry));

        assertTrue(findSync(PROJECT_ID, UID, AclUidType.USER).isPresent());
        assertFalse(findSync(PROJECT_ID2, UID, AclUidType.USER).isPresent());
        assertFalse(findSync(PROJECT_ID, UID + 1, AclUidType.USER).isPresent());
        assertFalse(findSync(PROJECT_ID, UID, AclUidType.GROUP).isPresent());
    }

    @Test
    public void delete() {
        assertTrue(createSync(projectAclEntry()));
        assertTrue(createSync(projectAclEntry(UID + 1)));
        assertTrue(createSync(projectAclEntry(UID + 2)));
        assertTrue(createSync(projectAclEntry(UID + 3)));
        assertTrue(createSync(projectAclEntry(UID + 4)));

        assertTrue(deleteSync(projectAclEntry(UID + 1)));
        assertFalse(deleteSync(projectAclEntry(UID + 11)));
        assertFalse(findSync(PROJECT_ID, UID + 1, AclUidType.USER).isPresent());
        assertTrue(findSync(PROJECT_ID, UID + 2, AclUidType.USER).isPresent());
    }

    @Test
    public void list() {
        assertTrue(createSync(projectAclEntry()));
        assertTrue(createSync(projectAclEntry(UID + 1)));
        assertTrue(createSync(projectAclEntry(UID + 2)));
        assertTrue(createSync(projectAclEntry(UID + 3)));
        assertTrue(createSync(new ProjectAclEntry(PROJECT_ID2, UID + 4, AclUidType.USER, ROLES, 0)));

        TokenBasePage<ProjectAclEntry> allSync = listSync(PROJECT_ID, 2, "");
        Assert.assertEquals(2, allSync.getItems().size());
        Assert.assertTrue(allSync.getItems().contains(projectAclEntry()));
        Assert.assertTrue(allSync.getItems().contains(projectAclEntry(UID + 1)));

        allSync = listSync(PROJECT_ID, 2, allSync.getNextPageToken());
        Assert.assertEquals(2, allSync.getItems().size());
        Assert.assertTrue(allSync.getItems().contains(projectAclEntry(UID + 2)));
        Assert.assertTrue(allSync.getItems().contains(projectAclEntry(UID + 3)));
        Assert.assertTrue(allSync.getNextPageToken().isEmpty());
    }

    @Test
    public void insertLostUpdate() {
        var expected = IntStream.range(0, 10)
                .mapToObj(ignore -> projectAclEntry())
                .collect(Collectors.toList());

       var result = expected.parallelStream()
                .map(this::createSync)
                .collect(Collectors.toList());
        assertEquals(1, result.stream().filter(aBoolean -> aBoolean).count());
    }

    private TokenBasePage<ProjectAclEntry> listSync(String projectId, int pageSize, String pageToken) {
        return join(getDao().list(projectId, pageSize, pageToken));
    }

    private boolean createSync(ProjectAclEntry projectAclEntry) {
        return join(getDao().create(projectAclEntry));
    }

    private boolean deleteSync(ProjectAclEntry projectAclEntry) {
        return join(getDao().delete(projectAclEntry));
    }

    private void updateSync(ProjectAclEntry projectAclEntry) {
        join(getDao().update(projectAclEntry));
    }

    private Optional<ProjectAclEntry> findSync(String projectId, String uid, AclUidType type) {
        return join(getDao().find(projectId, uid, type));
    }

    private ProjectAclEntry projectAclEntry() {
        return new ProjectAclEntry(PROJECT_ID, UID, AclUidType.USER, ROLES, 0);
    }

    private ProjectAclEntry projectAclEntry(String uid) {
        return new ProjectAclEntry(PROJECT_ID, uid, AclUidType.USER, ROLES, 0);
    }
}
