package ru.yandex.solomon.acl.db;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractServiceProviderAclEntryDaoTest {

    private static final String SERVICE_PROVIDER_ID = "sp 1";
    private static final String SERVICE_PROVIDER_ID2 = "sp 2";
    private static final String UID = "some uid";
    private static final Set<String> ROLES = Set.of("role 1", "role 2", "role 3");

    protected abstract ServiceProviderAclEntryDao getDao();

    @Test
    public void create() {
        assertTrue(createSync(entry()));
    }

    @Test
    public void update() {
        ServiceProviderAclEntry entry = entry();
        assertTrue(createSync(entry));

        entry.setRoles(Set.of("role 4"));
        updateSync(entry);
        var actual = findSync(SERVICE_PROVIDER_ID, UID, AclUidType.USER);
        assertTrue(actual.isPresent());
        assertEquals(SERVICE_PROVIDER_ID, actual.get().getServiceProviderId());
        assertEquals(UID, actual.get().getUid());
        assertEquals(Set.of("role 4"), actual.get().getRoles());
        assertEquals(AclUidType.USER, actual.get().getType());
    }

    @Test(expected = CompletionException.class)
    public void concurrentActions() {
        assertTrue(createSync(entry()));

        //now version 1
        updateSync(entry());
        Optional<ServiceProviderAclEntry> entry = findSync(SERVICE_PROVIDER_ID, UID, AclUidType.USER);
        //update to version 2
        updateSync(entry.get());
        assertFalse(deleteSync(entry.get()));

        //update with version 1, but actual version 2
        updateSync(entry.get());
    }

    @Test
    public void find() {
        ServiceProviderAclEntry entry = entry();
        assertTrue(createSync(entry));

        assertTrue(findSync(SERVICE_PROVIDER_ID, UID, AclUidType.USER).isPresent());
        assertFalse(findSync(SERVICE_PROVIDER_ID2, UID, AclUidType.USER).isPresent());
        assertFalse(findSync(SERVICE_PROVIDER_ID, UID + 1, AclUidType.USER).isPresent());
        assertFalse(findSync(SERVICE_PROVIDER_ID, UID, AclUidType.GROUP).isPresent());
    }

    @Test
    public void delete() {
        assertTrue(createSync(entry()));
        assertTrue(createSync(entry(UID + 1)));
        assertTrue(createSync(entry(UID + 2)));
        assertTrue(createSync(entry(UID + 3)));
        assertTrue(createSync(entry(UID + 4)));

        assertTrue(deleteSync(entry(UID + 1)));
        assertFalse(deleteSync(entry(UID + 11)));
        assertFalse(findSync(SERVICE_PROVIDER_ID, UID + 1, AclUidType.USER).isPresent());
        assertTrue(findSync(SERVICE_PROVIDER_ID, UID + 2, AclUidType.USER).isPresent());
    }

    @Test
    public void list() {
        assertTrue(createSync(entry()));
        assertTrue(createSync(entry(UID + 1)));
        assertTrue(createSync(entry(UID + 2)));
        assertTrue(createSync(entry(UID + 3)));
        assertTrue(createSync(new ServiceProviderAclEntry(SERVICE_PROVIDER_ID2, UID + 4, AclUidType.USER, ROLES, 0)));

        TokenBasePage<ServiceProviderAclEntry> allSync = listSync(SERVICE_PROVIDER_ID, 2, "");
        Assert.assertEquals(2, allSync.getItems().size());
        Assert.assertTrue(allSync.getItems().contains(entry()));
        Assert.assertTrue(allSync.getItems().contains(entry(UID + 1)));

        allSync = listSync(SERVICE_PROVIDER_ID, 2, allSync.getNextPageToken());
        Assert.assertEquals(2, allSync.getItems().size());
        Assert.assertTrue(allSync.getItems().contains(entry(UID + 2)));
        Assert.assertTrue(allSync.getItems().contains(entry(UID + 3)));
        Assert.assertTrue(allSync.getNextPageToken().isEmpty());
    }

    @Test
    public void insertLostUpdate() {
        var expected = IntStream.range(0, 10)
                .mapToObj(ignore -> entry())
                .collect(Collectors.toList());

        var result = expected.parallelStream()
                .map(this::createSync)
                .collect(Collectors.toList());
        assertEquals(1, result.stream().filter(aBoolean -> aBoolean).count());
    }

    private TokenBasePage<ServiceProviderAclEntry> listSync(String serviceProviderId, int pageSize, String pageToken) {
        return join(getDao().list(serviceProviderId, pageSize, pageToken));
    }

    private boolean createSync(ServiceProviderAclEntry entry) {
        return join(getDao().create(entry));
    }

    private boolean deleteSync(ServiceProviderAclEntry entry) {
        return join(getDao().delete(entry));
    }

    private void updateSync(ServiceProviderAclEntry entry) {
        join(getDao().update(entry));
    }

    private Optional<ServiceProviderAclEntry> findSync(String serviceProviderId, String uid, AclUidType type) {
        return join(getDao().find(serviceProviderId, uid, type));
    }

    private ServiceProviderAclEntry entry() {
        return new ServiceProviderAclEntry(SERVICE_PROVIDER_ID, UID, AclUidType.USER, ROLES, 0);
    }

    private ServiceProviderAclEntry entry(String uid) {
        return new ServiceProviderAclEntry(SERVICE_PROVIDER_ID, uid, AclUidType.USER, ROLES, 0);
    }
}
