package ru.yandex.solomon.acl.db;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractSystemAclEntryDaoTest {

    private static final String UID = "some uid";
    private static final Set<String> ROLES = Set.of("role 1", "role 2", "role 3");

    protected abstract SystemAclEntryDao getDao();

    @Test
    public void create() {
        assertTrue(createSync(systemAclEntry()));
    }

    @Test
    public void update() {
        SystemAclEntry systemAclEntry = systemAclEntry();
        assertTrue(createSync(systemAclEntry));

        systemAclEntry.setRoles(Set.of("role 4"));
        updateSync(systemAclEntry);
        var actual = findSync(UID, AclUidType.USER);
        assertTrue(actual.isPresent());
        assertEquals(UID, actual.get().getUid());
        assertEquals(Set.of("role 4"), actual.get().getRoles());
        assertEquals(AclUidType.USER, actual.get().getType());
    }

    @Test(expected = CompletionException.class)
    public void concurrentActions() {
        assertTrue(createSync(systemAclEntry()));

        //now version 1
        updateSync(systemAclEntry());
        Optional<SystemAclEntry> systemAclEntry = findSync(UID, AclUidType.USER);
        //update to version 2
        updateSync(systemAclEntry.get());
        assertFalse(deleteSync(systemAclEntry.get()));

        //update with version 1, but actual version 2
        updateSync(systemAclEntry.get());
    }

    @Test
    public void find() {
        assertTrue(createSync(systemAclEntry()));

        assertTrue(findSync(UID, AclUidType.USER).isPresent());
        assertFalse(findSync(UID + 1, AclUidType.USER).isPresent());
        assertFalse(findSync(UID, AclUidType.GROUP).isPresent());
    }

    @Test
    public void delete() {
        assertTrue(createSync(systemAclEntry()));
        assertTrue(createSync(systemAclEntry(UID + 1)));
        assertTrue(createSync(systemAclEntry(UID + 2)));
        assertTrue(createSync(systemAclEntry(UID + 3)));
        assertTrue(createSync(systemAclEntry(UID + 4)));

        assertTrue(deleteSync(systemAclEntry(UID + 1)));
        assertFalse(deleteSync(systemAclEntry(UID + 11)));
        assertFalse(findSync(UID + 1, AclUidType.USER).isPresent());
        assertTrue(findSync(UID + 2, AclUidType.USER).isPresent());
    }

    private boolean createSync(SystemAclEntry systemAclEntry) {
        return join(getDao().create(systemAclEntry));
    }

    private boolean deleteSync(SystemAclEntry systemAclEntry) {
        return join(getDao().delete(systemAclEntry));
    }

    private void updateSync(SystemAclEntry systemAclEntry) {
        join(getDao().update(systemAclEntry));
    }

    private Optional<SystemAclEntry> findSync(String uid, AclUidType type) {
        return join(getDao().find(uid, type));
    }

    private SystemAclEntry systemAclEntry() {
        return new SystemAclEntry(UID, AclUidType.USER, ROLES, 0);
    }

    private SystemAclEntry systemAclEntry(String uid) {
        return new SystemAclEntry(uid, AclUidType.USER, ROLES, 0);
    }
}
