package ru.yandex.solomon.alert.client.grpc;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.common.net.HostAndPort;
import com.google.protobuf.Message;
import io.grpc.Status;

import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingCluster implements AutoCloseable {
    private static final int MAX_RETRY = 5;
    private static final long RETRY_INITIAL_DELAY = 100;
    private final List<GrpcTransport> nodes;
    private final AtomicLong nodeCursor = new AtomicLong(0);
    @Nullable
    private final ScheduledExecutorService timer;
    private final MetricRegistry registry;

    public AlertingCluster(List<HostAndPort> addresses, GrpcClientOptions options) {
        this.nodes = addresses.stream()
                .map(address -> new GrpcTransport(address, options))
                .collect(Collectors.toList());
        this.timer = options.getTimer().orElse(null);
        this.registry = options.getMetricRegistry();
    }

    public <ReqT extends Message, RespT> CompletableFuture<RespT> execute(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request) {
        return execute(endpoint, request, endpoint.getDeadline(request));
    }

    public <ReqT extends Message, RespT> CompletableFuture<RespT> execute(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request, long deadlineMillis) {
        return new EndpointCall<>(endpoint, request, deadlineMillis).call();
    }

    @Nullable
    private GrpcTransport chooseNode() {
        int shift = Math.toIntExact(nodeCursor.getAndIncrement() % (long) nodes.size());
        GrpcTransport firstConnected = null;
        for (int index = 0; index < nodes.size(); index++) {
            int actualIndex = (index + shift) % nodes.size();
            var node = nodes.get(actualIndex);
            if (node.isReady()) {
                return node;
            } else if (node.isConnected()) {
                firstConnected = node;
            }
        }

        // not found ready nodes, choose first connected, maybe it will be ready
        return firstConnected;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("nodes", nodes)
                .toString();
    }

    @Override
    public void close() {
        nodes.forEach(GrpcTransport::close);
    }

    private class EndpointCall<ReqT, RespT> {
        private final EndpointDescriptor<ReqT, RespT> endpoint;
        private final CompletableFuture<RespT> doneFuture = new CompletableFuture<>();
        private final ReqT request;
        private final long deadlineMillis;
        private int attempt = 0;
        private CompletableFuture<RespT> lastResponse;

        public EndpointCall(EndpointDescriptor<ReqT, RespT> endpoint, ReqT request, long deadlineMillis) {
            this.endpoint = endpoint;
            this.request = request;
            this.deadlineMillis = deadlineMillis;
        }

        public CompletableFuture<RespT> call() {
            tryCall();
            return doneFuture;
        }

        private void tryCall() {
            GrpcTransport client = chooseNode();
            if (client == null) {
                lastResponse = CompletableFuture.failedFuture(Status.UNAVAILABLE
                        .withDescription("Not found ready node")
                        .asRuntimeException());
            } else {
                lastResponse = client.unaryCall(endpoint.getMethod(), request, deadlineMillis);
            }

            lastResponse.whenComplete(this::whenComplete);
        }

        private void whenComplete(RespT response, @Nullable Throwable e) {
            if (e != null) {
                Status status = Status.fromThrowable(e);
                if (status.getCode() == Status.Code.UNAVAILABLE) {
                    scheduleRetry();
                } else {
                    doneFuture.completeExceptionally(e);
                }
                return;
            }

            var status = endpoint.getStatusCode(response);
            registry.rate("alerting.client.call.status",
                    Labels.of("endpoint", endpoint.getMethod().getFullMethodName(),
                            "code", status.name()))
                    .inc();
            switch (status) {
                case SHARD_NOT_INITIALIZED:
                case NODE_UNAVAILABLE:
                    scheduleRetry();
                default:
                    doneFuture.complete(response);
            }
        }

        private void scheduleRetry() {
            if (attempt == 0) {
                attempt++;
                this.tryCall();
                return;
            }

            if (timer == null || attempt > MAX_RETRY) {
                CompletableFutures.whenComplete(lastResponse, doneFuture);
                return;
            }

            long delay = RETRY_INITIAL_DELAY * Math.round(Math.pow(2, attempt++));
            long jitter = ThreadLocalRandom.current().nextLong(delay / 4);
            timer.schedule(this::tryCall, delay + jitter, TimeUnit.MILLISECONDS);
        }
    }
}
