package ru.yandex.solomon.alert.client.grpc;

import java.util.function.Function;
import java.util.function.ToLongFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.MethodDescriptor;

import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class EndpointDescriptor<ReqT, RespT> {
    private final MethodDescriptor<ReqT, RespT> method;
    private final ToLongFunction<ReqT> deadlineFn;
    private final Function<RespT, ERequestStatusCode> responseStatusFn;

    private EndpointDescriptor(Builder<ReqT, RespT> builder) {
        this.method = requireNonNull(builder.method);
        this.deadlineFn = requireNonNull(builder.deadlineFn);
        this.responseStatusFn = requireNonNull(builder.responseStatusFn);
    }

    public static <ReqT, RespT> Builder<ReqT, RespT> newBuilder() {
        return new Builder<>();
    }

    public MethodDescriptor<ReqT, RespT> getMethod() {
        return method;
    }

    public long getDeadline(ReqT request) {
        return deadlineFn.applyAsLong(request);
    }

    public ERequestStatusCode getStatusCode(RespT response) {
        return responseStatusFn.apply(response);
    }

    public static class Builder<ReqT, RespT> {
        private MethodDescriptor<ReqT, RespT> method;
        private ToLongFunction<ReqT> deadlineFn;
        private Function<RespT, ERequestStatusCode> responseStatusFn;

        public Builder() {
        }

        public Builder<ReqT, RespT> setGrpcMethod(MethodDescriptor<ReqT, RespT> method) {
            this.method = method;
            return this;
        }

        public Builder<ReqT, RespT> setDeadlineResolver(ToLongFunction<ReqT> deadlineFn) {
            this.deadlineFn = deadlineFn;
            return this;
        }

        public Builder<ReqT, RespT> setStatusCodeResolver(Function<RespT, ERequestStatusCode> responseStatusFn) {
            this.responseStatusFn = responseStatusFn;
            return this;
        }

        public EndpointDescriptor<ReqT, RespT> build() {
            return new EndpointDescriptor<>(this);
        }
    }
}
