package ru.yandex.solomon.alert.client.stub;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TAlertList;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListAlert;
import ru.yandex.solomon.alert.protobuf.TListAlertList;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;

import static java.util.concurrent.CompletableFuture.supplyAsync;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.OK;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertApiStub extends AbstractAlertApi {
    private final ConcurrentHashMap<Key, TAlert> alerts = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<TCreateAlertResponse> createAlert(TCreateAlertRequest request) {
        return supplyAsync(() -> {
            TAlert alert = request.getAlert();
            var key = new Key(alert.getProjectId(), alert.getId());
            var prev = alerts.putIfAbsent(key, request.getAlert());
            if (prev != null) {
                return TCreateAlertResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Alert with id " + key + " already exists")
                        .build();
            }

            return TCreateAlertResponse.newBuilder()
                    .setRequestStatus(OK)
                    .setAlert(alert)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TReadAlertResponse> readAlert(TReadAlertRequest request) {
        return supplyAsync(() -> {
            var key = new Key(request.getProjectId(), request.getAlertId());
            TAlert alert = alerts.get(key);
            if (alert == null) {
                return TReadAlertResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.NOT_FOUND)
                        .setStatusMessage("Alert with id " + key + " not found")
                        .build();
            }

            return TReadAlertResponse.newBuilder()
                    .setRequestStatus(OK)
                    .setAlert(alert)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TReadAlertInterpolatedResponse> readAlert(TReadAlertInterpolatedRequest request) {
        return supplyAsync(() -> {
            var key = new Key(request.getProjectId(), request.getAlertId());
            TAlert alert = alerts.get(key);
            if (alert == null) {
                return TReadAlertInterpolatedResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.NOT_FOUND)
                        .setStatusMessage("Alert with id " + key + " not found")
                        .build();
            }

            return TReadAlertInterpolatedResponse.newBuilder()
                    .setRequestStatus(OK)
                    .setAlert(alert)
                    .setInterpolatedAlert(alert)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TUpdateAlertResponse> updateAlert(TUpdateAlertRequest request) {
        return supplyAsync(() -> {
            TAlert alert = request.getAlert();
            var key = new Key(alert.getProjectId(), alert.getId());
            var prev = alerts.get(key);
            if (prev == null) {
                return TUpdateAlertResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Alert with id " + key + " not found")
                        .build();
            }

            if (alerts.replace(key, prev, alert)) {
                return TUpdateAlertResponse.newBuilder()
                        .setRequestStatus(OK)
                        .setAlert(alert)
                        .build();
            }

            return TUpdateAlertResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.CONCURRENT_MODIFICATION)
                    .setStatusMessage("Concurrent update alert " + key)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TDeleteAlertResponse> deleteAlert(TDeleteAlertRequest request) {
        return supplyAsync(() -> {
            var key = new Key(request.getProjectId(), request.getAlertId());
            var prev = alerts.get(key);
            if (prev == null) {
                return TDeleteAlertResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Alert with id " + key + " not found")
                        .build();
            }

            if (alerts.remove(key, prev)) {
                return TDeleteAlertResponse.newBuilder()
                        .setRequestStatus(OK)
                        .build();
            }

            return TDeleteAlertResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.CONCURRENT_MODIFICATION)
                    .setStatusMessage("Concurrent update alert " + key)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TListAlertResponse> listAlerts(TListAlertRequest request) {
        return supplyAsync(() -> {
            int offset = pageOffset(request.getPageToken());
            int limit = pageLimit(request.getPageSize());

            var filter = filterByProject(request.getProjectId())
                    .and(filterByFolderId(request.getFolderId())
                    .and(filterByName(request.getFilterByName())));

            if (request.getFullResultModel()) {
                var alertList = alerts.values()
                        .stream()
                        .filter(filter)
                        .skip(offset)
                        .limit(limit)
                        .collect(Collectors.toList());

                return TListAlertResponse.newBuilder()
                        .setRequestStatus(OK)
                        .setAlertList(TAlertList.newBuilder().addAllAlerts(alertList).build())
                        .setNextPageToken(nextPageToken(offset, limit, alertList.size()))
                        .build();
            }
            var alertList = alerts.values()
                    .stream()
                    .filter(filter)
                    .skip(offset)
                    .limit(limit)
                    .map(alert -> TListAlert.newBuilder()
                            .setId(alert.getId())
                            .setProjectId(alert.getProjectId())
                            .setName(alert.getName())
                            .setAlertType(alert.getTypeCase() == TAlert.TypeCase.THRESHOLD
                                    ? EAlertType.THRESHOLD
                                    : EAlertType.EXPRESSION)
                            .build())
                    .collect(Collectors.toList());

            return TListAlertResponse.newBuilder()
                    .setRequestStatus(OK)
                    .addAllAlerts(alertList)
                    .setListAlertList(TListAlertList.newBuilder().addAllAlerts(alertList).build())
                    .setNextPageToken(nextPageToken(offset, limit, alertList.size()))
                    .build();
        });
    }

    @Override
    public CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request) {
        if (request.getProjectId().equals("fail")) {
             return CompletableFuture.failedFuture(new IllegalArgumentException("Failed project"));
        }
        return supplyAsync(() -> UpdateAlertTemplateVersionResponse.newBuilder()
                .setUpdated(request.getProjectId().length())
                .build());
    }

    private Predicate<TAlert> filterByProject(String projectId) {
        return alert -> projectId.isEmpty() || projectId.equals(alert.getProjectId());
    }

    private Predicate<TAlert> filterByFolderId(String folderId) {
        return alert -> folderId.isEmpty() || folderId.equals(alert.getFolderId());
    }

    private Predicate<TAlert> filterByName(String name) {
        return alert -> name.isEmpty() || StringUtils.containsIgnoreCase(alert.getName(), name);
    }

    private int pageOffset(String token) {
        return token.isEmpty() ? 0 : Integer.parseInt(token);
    }

    private int pageLimit(int limit) {
        return limit == 0 ? Integer.MAX_VALUE : limit;
    }

    private String nextPageToken(int offset, int limit, int size) {
        if (size < limit) {
            return "";
        }

        return Integer.toString(offset + size);
    }

    private static record Key(String projectId, String alertId) {
    }

    @Override
    public CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(CreateAlertsFromTemplateRequest request) {
        return CompletableFuture.completedFuture(CreateAlertsFromTemplateResponse
                .newBuilder()
                .setRequestStatusCode(OK)
                .build());
    }
}
