package ru.yandex.solomon.alert.client.stub;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;

import static java.util.concurrent.CompletableFuture.supplyAsync;


/**
 * @author Vladimir Gordiychuk
 */
public class NotificationApiStub extends AbstractNotificationApi {
    private final ConcurrentHashMap<Key, TNotification> notifications = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<TCreateNotificationResponse> createNotification(TCreateNotificationRequest request) {
        return supplyAsync(() -> {
            var notification = request.getNotification();
            var key = new Key(notification.getProjectId(), notification.getId());
            var prev = notifications.putIfAbsent(key, request.getNotification());
            if (prev != null) {
                return TCreateNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Notification with id " + key + " already exists")
                        .build();
            }

            return TCreateNotificationResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.OK)
                    .setNotification(notification)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TReadNotificationResponse> readNotification(TReadNotificationRequest request) {
        return supplyAsync(() -> {
            var key = new Key(request.getProjectId(), request.getNotificationId());
            TNotification notification = notifications.get(key);
            if (notification == null) {
                return TReadNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.NOT_FOUND)
                        .setStatusMessage("Notification with id " + key + " not found")
                        .build();
            }

            return TReadNotificationResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.OK)
                    .setNotification(notification)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TUpdateNotificationResponse> updateNotification(TUpdateNotificationRequest request) {
        return supplyAsync(() -> {
            TNotification notification = request.getNotification();
            var key = new Key(notification.getProjectId(), notification.getId());
            var prev = notifications.get(key);
            if (prev == null) {
                return TUpdateNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Notification with id " + key + " not found")
                        .build();
            }

            if (notifications.replace(key, prev, notification)) {
                return TUpdateNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.OK)
                        .setNotification(notification)
                        .build();
            }

            return TUpdateNotificationResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.CONCURRENT_MODIFICATION)
                    .setStatusMessage("Concurrent update notification " + key)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TDeleteNotificationResponse> deleteNotification(TDeleteNotificationRequest request) {
        return supplyAsync(() -> {
            var key = new Key(request.getProjectId(), request.getNotificationId());
            var prev = notifications.get(key);
            if (prev == null) {
                return TDeleteNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                        .setStatusMessage("Notification with id " + key + " not found")
                        .build();
            }

            if (notifications.remove(key, prev)) {
                return TDeleteNotificationResponse.newBuilder()
                        .setRequestStatus(ERequestStatusCode.OK)
                        .build();
            }

            return TDeleteNotificationResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.CONCURRENT_MODIFICATION)
                    .setStatusMessage("Concurrent update notification " + key)
                    .build();
        });
    }

    @Override
    public CompletableFuture<TListNotificationsResponse> listNotification(TListNotificationsRequest request) {
        return supplyAsync(() -> {
            int offset = pageOffset(request.getPageToken());
            int limit = pageLimit(request.getPageSize());

            var filter = filterByProject(request.getProjectId())
                    .and(filterByFolderId(request.getFolderId())
                    .and(filterByName(request.getFilterByName())));

            var notificationList = notifications.values()
                    .stream()
                    .filter(filter)
                    .skip(offset)
                    .limit(limit)
                    .collect(Collectors.toList());

            return TListNotificationsResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.OK)
                    .addAllNotification(notificationList)
                    .setNextPageToken(nextPageToken(offset, limit, notificationList.size()))
                    .build();
        });
    }

    private Predicate<TNotification> filterByProject(String projectId) {
        return notification -> projectId.isEmpty() || projectId.equals(notification.getProjectId());
    }

    private Predicate<TNotification> filterByFolderId(String folderId) {
        return notification -> folderId.isEmpty() || folderId.equals(notification.getFolderId());
    }

    private Predicate<TNotification> filterByName(String name) {
        return notification -> name.isEmpty() || StringUtils.containsIgnoreCase(notification.getName(), name);
    }

    private int pageOffset(String token) {
        return token.isEmpty() ? 0 : Integer.parseInt(token);
    }

    private int pageLimit(int limit) {
        return limit == 0 ? Integer.MAX_VALUE : limit;
    }

    private String nextPageToken(int offset, int limit, int size) {
        if (size < limit) {
            return "";
        }

        return Integer.toString(offset + size);
    }

    private static record Key(String projectId, String notificationId) {
    }
}
