package ru.yandex.solomon.alert.client;

import java.util.Arrays;
import java.util.List;

import com.google.common.net.HostAndPort;
import io.grpc.BindableService;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import io.grpc.testing.GrpcServerRulePublic;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.solomon.alert.protobuf.TAlertServiceGrpc;
import ru.yandex.solomon.alert.protobuf.TEvaluationState;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingClientTest {
    @Rule
    public final GrpcServerRulePublic aliceServer = new GrpcServerRulePublic();

    @Rule
    public final GrpcServerRulePublic bobServer = new GrpcServerRulePublic();


    private ManualClock clock;
    private AlertingClient client;
    private ManualScheduledExecutorService timer;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(2, clock);
        var options = DefaultClientOptions.newBuilder()
                .setChannelFactory(new InProcessChannelFactory())
                .setTimer(timer)
                .build();

        List<HostAndPort> addresses = Arrays.asList(
                HostAndPort.fromHost(aliceServer.getServerName()),
                HostAndPort.fromHost(bobServer.getServerName()));

        client = AlertingClients.create(addresses, options);
    }

    @After
    public void tearDown() throws Exception {
        if (timer != null) {
            timer.shutdownNow();
        }
    }

    @Test
    public void unimplementedCall() {
        Status status = client.readAlert(TReadAlertRequest.newBuilder()
                .setProjectId("junk")
                .setAlertId("my-alert-id")
                .build())
                .thenApply(response -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertThat(status.getCode(), equalTo(Status.Code.UNIMPLEMENTED));
    }

    @Test
    public void successCall() {
        BindableService service = new TAlertServiceGrpc.TAlertServiceImplBase() {
            @Override
            public void readEvaluationState(TReadEvaluationStateRequest request, StreamObserver<TReadEvaluationStateResponse> responseObserver) {
                responseObserver.onNext(TReadEvaluationStateResponse.newBuilder()
                        .setState(TEvaluationState.newBuilder()
                                .setProjectId("junk")
                                .setAlertId("my-alert-id")
                                .setAlertVersion(42)
                                .build())
                        .build());
                responseObserver.onCompleted();
            }
        };

        aliceServer.getServiceRegistry().addService(service);
        bobServer.getServiceRegistry().addService(service);

        Status status = client.readEvaluationState(TReadEvaluationStateRequest.newBuilder()
                .setProjectId("junk")
                .setAlertId("my-alert-id")
                .build())
                .thenApply(response -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertThat(status.getCode(), equalTo(Status.Code.OK));
    }

    @Test
    public void retryCallWhenServiceUnavailable() {
        BindableService service = new TAlertServiceGrpc.TAlertServiceImplBase() {
            @Override
            public void readEvaluationState(TReadEvaluationStateRequest request, StreamObserver<TReadEvaluationStateResponse> responseObserver) {
                responseObserver.onNext(TReadEvaluationStateResponse.newBuilder()
                        .setState(TEvaluationState.newBuilder()
                                .setProjectId("junk")
                                .setAlertId("my-alert-id")
                                .setAlertVersion(42)
                                .build())
                        .build());
                responseObserver.onCompleted();
            }
        };

        aliceServer.getServiceRegistry().addService(service);
        bobServer.after();

        TReadEvaluationStateRequest request = TReadEvaluationStateRequest.newBuilder()
                .setProjectId("junk")
                .setAlertId("my-alert-id")
                .build();

        for (int index = 0; index < 10; index++) {
            Status status = client.readEvaluationState(request)
                    .thenApply(response -> Status.OK)
                    .exceptionally(Status::fromThrowable)
                    .join();

            assertThat(status.getCode(), equalTo(Status.Code.OK));
        }
    }
}
