package ru.yandex.solomon.auth;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.apache.commons.lang3.tuple.Pair;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import yandex.cloud.auth.api.AsyncCloudAuthClient;
import yandex.cloud.auth.api.CloudAuthConstants;
import yandex.cloud.auth.grpc.internal.AsyncCloudAuthGrpcClientImpl;

import ru.yandex.abc.AbcClient;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxClientOptions;
import ru.yandex.blackbox.BlackboxClients;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.session.SessionClient;
import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.passport.tvmauth.BlackboxEnv;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.auth.authorizers.AbcAuthorizer;
import ru.yandex.solomon.auth.authorizers.InternalIamAuthorizer;
import ru.yandex.solomon.auth.authorizers.ProjectAclAuthorizer;
import ru.yandex.solomon.auth.authorizers.ProjectAclOrIamAuthorizer;
import ru.yandex.solomon.auth.authorizers.ProjectManagerAuthorizer;
import ru.yandex.solomon.auth.authorizers.RoleAuthorizer;
import ru.yandex.solomon.auth.authorizers.ServiceProviderAuthorizer;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.iam.ApiKeyAuthenticator;
import ru.yandex.solomon.auth.iam.FakeCloudAuthClient;
import ru.yandex.solomon.auth.iam.IamAuthenticator;
import ru.yandex.solomon.auth.iam.IamAuthorizer;
import ru.yandex.solomon.auth.iam.IamTokenContext;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.auth.local.AsUserAuthenticator;
import ru.yandex.solomon.auth.oauth.OAuthAuthenticator;
import ru.yandex.solomon.auth.openid.OpenIdAuthenticator;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthenticator;
import ru.yandex.solomon.auth.tvm.TvmAuthenticator;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.ProjectManagerConfig;
import ru.yandex.solomon.config.protobuf.StaffClientConfig;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig.TIamConfig;
import ru.yandex.solomon.config.thread.LazyThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Sergey Polovko
 */
@Import(IamTokenContext.class)
public class AuthContext {

    @Bean
    HttpAuthenticator httpAuthenticator(Authenticator authenticator) {
        return new HttpAuthenticator(authenticator);
    }

    @Bean
    ApiKeyAuthenticator apiKeyAuthenticator(AsyncCloudAuthClient cloudAuthClient) {
        return new ApiKeyAuthenticator(cloudAuthClient);
    }

    @Bean
    Authenticator authenticator(
        TAuthConfig authConfig,
        BlackboxClient blackboxClient,
        AsyncCloudAuthClient cloudAuthClient,
        Optional<TvmClient> tvmClient,
        Optional<SessionClient> sessionClientO)
    {
        List<Pair<AuthType, Authenticator>> authenticators = new ArrayList<>();

        if (authConfig.hasOAuthConfig()) {
            authenticators.add(Pair.of(AuthType.OAuth, new OAuthAuthenticator(blackboxClient)));
        }

        if (authConfig.hasSessionIdConfig()) {
            authenticators.add(Pair.of(AuthType.SessionIdCookie, new SessionIdAuthenticator(blackboxClient)));
        }

        if (authConfig.hasTvmConfig()) {
            authenticators.add(Pair.of(AuthType.TvmService, new TvmAuthenticator(tvmClient.orElseThrow(), blackboxClient)));
            authenticators.add(Pair.of(AuthType.TvmUser, new TvmAuthenticator(tvmClient.orElseThrow(), blackboxClient)));
        }

        if (authConfig.hasIamConfig()) {
            authenticators.add(Pair.of(AuthType.IAM, new IamAuthenticator(cloudAuthClient)));
        }

        if (authConfig.hasAsUserConfig()) {
            authenticators.add(Pair.of(AuthType.AsUser, new AsUserAuthenticator()));
        }

        // XXX: OpenIdAuthenticator must be the last in authenticators list
        if (authConfig.hasOpenIdConfig()) {
            var sessionClient = sessionClientO.orElseThrow(() -> {
                String message = "OpenId auth method cannot be used without configuration of SessionClient";
                return new RuntimeException(message);
            });
            var openIdConfig = authConfig.getOpenIdConfig();
            var authenticator = new OpenIdAuthenticator(
                    openIdConfig.getDomain(),
                    openIdConfig.getClientId(),
                    openIdConfig.getFederationId(),
                    sessionClient);
            authenticators.add(Pair.of(AuthType.OpenId, authenticator));
        }

        if (authenticators.isEmpty()) {
            throw new IllegalStateException("AuthConfig is empty or absent");
        }

        return Authenticator.cache(Authenticator.mux(authenticators));
    }

    @Bean
    TvmClient tvmClient(TAuthConfig authConfig, SecretProvider secrets) {
        if (!authConfig.hasTvmConfig()) {
            return null;
        }
        TAuthConfig.TTvmConfig tvmConfig = authConfig.getTvmConfig();
        Optional<String> clientSecret = secrets.getSecret(tvmConfig.getSecret());
        try (TvmApiSettings settings = TvmApiSettings.create()
                .setSelfTvmId(tvmConfig.getClientId())
                .enableServiceTicketChecking()
                .enableUserTicketChecking(BlackboxEnv.PROD_YATEAM))
        {
            clientSecret.ifPresent(s -> settings.enableServiceTicketsFetchOptions(s, Ints.toArray(tvmConfig.getDestinationIdsList())));
            return NativeTvmClient.create(settings);
        }
    }

    @Bean
    BlackboxClient blackboxClient(TAuthConfig authConfig) {
        if (authConfig.hasOAuthConfig()) {
            TAuthConfig.TOAuthConfig oauthConfig = authConfig.getOAuthConfig();
            var opts = BlackboxClientOptions.newBuilder()
                    .setUrl("http://blackbox.yandex-team.ru")
                    .setHost(oauthConfig.getHost())
                    .build();
            return BlackboxClients.create(new DefaultAsyncHttpClient(), opts);
        }
        return BlackboxClients.fake();
    }

    @Bean
    AsyncCloudAuthClient cloudAuthClient(LazyThreadPoolProvider threads, MetricRegistry registry, TAuthConfig authConfig) {
        if (!authConfig.hasIamConfig()) {
            return new FakeCloudAuthClient();
        }

        TAuthConfig.TIamConfig iamConfig = authConfig.getIamConfig();

        String threadPoolName = iamConfig.getThreadPoolName();
        Executor threadPool = !threadPoolName.isEmpty()
                ? threads.getExecutorService(threadPoolName, "AuthConfig.IamConfig.ThreadPool")
                : MoreExecutors.directExecutor();

        Duration readTimeout = TimeConverter.protoToDuration(iamConfig.getReadTimeout(), Duration.ofSeconds(2));
        HostAndPort address = HostAndPort.fromString(authConfig.getIamConfig().getAccessServiceAddresses());
        ManagedChannel channel = ManagedChannelBuilder.forAddress(address.getHost(), address.getPort())
                .userAgent("Solomon")
                .keepAliveTime(CloudAuthConstants.DEFAULT_KEEP_ALIVE_TIME.toNanos(), TimeUnit.NANOSECONDS)
                .keepAliveTimeout(CloudAuthConstants.DEFAULT_KEEP_ALIVE_TIMEOUT.toNanos(), TimeUnit.NANOSECONDS)
                .keepAliveWithoutCalls(true)
                .enableRetry()
                .intercept(new MetricClientInterceptor(HostUtils.getFqdn(), registry))
                .maxRetryAttempts(CloudAuthConstants.DEFAULT_RETRIES)
                .build();

        return new AsyncCloudAuthGrpcClientImpl(channel, readTimeout, threadPool);
    }

    @Bean
    @Nullable
    InternalAuthorizer internalAuthorizer(TAuthConfig authConfig) {
        if (SolomonEnv.DEVELOPMENT.isActive()) {
            return new InternalAuthorizer(Set.of()) {
                @Override
                public CompletableFuture<Account> authorize(AuthSubject subject) {
                    return CompletableFuture.completedFuture(Account.ANONYMOUS);
                }
            };
        }
        if (!authConfig.hasInternalAccess()) {
            return null;
        }
        var allowed = Set.copyOf(authConfig.getInternalAccess().getAllowList());
        return new InternalAuthorizer(allowed);
    }

    @Bean
    Authorizer authorizer(
            SolomonConfHolder confHolder,
            MetricRegistry registry,
            ThreadPoolProvider threads,
            Optional<ProjectAclEntryDao> projectAclEntryDao,
            Optional<SystemAclEntryDao> systemAclEntryDao,
            Optional<ServiceProviderAclEntryDao> serviceProviderAclEntryDao,
            Optional<GroupMemberDao> groupMemberDao,
            ServiceProvidersDao serviceProvidersDao,
            AsyncCloudAuthClient cloudAuthClient,
            TAuthConfig authConfig,
            Optional<TGatewayCloudConfig> gatewayCloudConfigOpt,
            Optional<TIdmConfig> idmConfig,
            Optional<StaffClientConfig> staffClientConfig,
            FeatureFlagsHolder featureFlagsHolder,
            Optional<InternalAuthorizer> internalAuthorizer,
            ProjectsDao projectsDao,
            Optional<ProjectManagerAuthorizer> projectManagerAuthorizer,
            Optional<AbcClient> abcClient)
    {
        List<Authorizer> authorizers = new ArrayList<>(2);
        boolean hasProjectAclAuth = authConfig.hasOAuthConfig() || authConfig.hasAsUserConfig();
        boolean hasIamAuth = authConfig.hasIamConfig();
        boolean isCloud = gatewayCloudConfigOpt.isPresent();
        boolean hasRoleAuth = idmConfig.isPresent() && staffClientConfig.isPresent();

        if (hasProjectAclAuth && hasIamAuth) {
            if (isCloud) {
                authorizers.add(createProjectAclOrIamAuthorizer(
                        confHolder,
                        cloudAuthClient,
                        authConfig.getIamConfig(),
                        gatewayCloudConfigOpt.get(),
                        featureFlagsHolder,
                        serviceProvidersDao));
            } else {
                if (hasRoleAuth) {
                    authorizers.add(createRoleAuthorizer(confHolder,
                            registry,
                            threads,
                            projectAclEntryDao.get(),
                            systemAclEntryDao.get(),
                            serviceProviderAclEntryDao.get(),
                            groupMemberDao.get(),
                            projectsDao,
                            serviceProvidersDao));
                } else {
                    authorizers.add(new ProjectAclAuthorizer(confHolder));
                }
                authorizers.add(new InternalIamAuthorizer(confHolder));
            }
        } else {
            if (hasRoleAuth && hasProjectAclAuth) {
                authorizers.add(createRoleAuthorizer(confHolder,
                        registry,
                        threads,
                        projectAclEntryDao.get(),
                        systemAclEntryDao.get(),
                        serviceProviderAclEntryDao.get(),
                        groupMemberDao.get(),
                        projectsDao,
                        serviceProvidersDao));
            } else if (hasProjectAclAuth) {
                authorizers.add(new ProjectAclAuthorizer(confHolder));
            }
            if (hasIamAuth) {
                var delegate = new IamAuthorizer(cloudAuthClient, authConfig.getIamConfig().getMaxInflight());
                authorizers.add(new ServiceProviderAuthorizer(delegate, confHolder, serviceProvidersDao));
            }
        }
        if (abcClient.isPresent()) {
            authorizers.add(new AbcAuthorizer(abcClient.get()));
        }

        Authorizer mux = Authorizer.mux(authorizers);
        if (projectManagerAuthorizer.isPresent()) {
            mux = Authorizer.oneOf(projectManagerAuthorizer.get(), mux);
        }
        if (internalAuthorizer.isPresent()) {
            mux = Authorizer.oneOf(internalAuthorizer.get().adapter(), mux);
        }


        return Authorizer.cache(mux);
    }

    @Bean
    public SolomonTeam.DisabledUsersManager disabledUsersManager() {
        return new SolomonTeam.DisabledUsersManager();
    }

    private static ProjectAclOrIamAuthorizer createProjectAclOrIamAuthorizer(
            SolomonConfHolder confHolder,
            AsyncCloudAuthClient cloudAuthClient,
            TIamConfig iamConfig,
            TGatewayCloudConfig gatewayCloudConfig,
            FeatureFlagsHolder featureFlagsHolder,
            ServiceProvidersDao serviceProvidersDao)
    {
        ProjectAclAuthorizer projectAclAuthorizer = new ProjectAclAuthorizer(confHolder);
        IamAuthorizer delegate = new IamAuthorizer(cloudAuthClient, iamConfig.getMaxInflight());
        return new ProjectAclOrIamAuthorizer(
                projectAclAuthorizer,
                new ServiceProviderAuthorizer(delegate, confHolder, serviceProvidersDao),
                gatewayCloudConfig,
                featureFlagsHolder);
    }

    private static RoleAuthorizer createRoleAuthorizer(
            SolomonConfHolder confHolder,
            MetricRegistry registry,
            ThreadPoolProvider threads,
            ProjectAclEntryDao projectAclEntryDao,
            SystemAclEntryDao systemAclEntryDao,
            ServiceProviderAclEntryDao serviceProviderAclEntryDao,
            GroupMemberDao groupMemberDao,
            ProjectsDao projectsDao,
            ServiceProvidersDao serviceProvidersDao)
    {
        return new RoleAuthorizer(confHolder,
                registry,
                threads,
                projectAclEntryDao,
                systemAclEntryDao,
                serviceProviderAclEntryDao,
                serviceProvidersDao,
                groupMemberDao,
                projectsDao);
    }

    @Bean
    public ProjectManagerAuthorizer projectManagerAuthorizer(
            Optional<ProjectManagerConfig> configOptional,
            @Qualifier("tvmClient") Optional<TvmClient> tvmClientOptional,
            @Qualifier("iamTokenProvider") Optional<TokenProvider> iamTokenProvider,
            FeatureFlagsHolder featureFlagsHolder,
            ClientOptionsFactory clientOptionsFactory)
    {
        if (configOptional.isEmpty()) {
            return null;
        }
        if (tvmClientOptional.isEmpty() && iamTokenProvider.isEmpty()) {
            return null;
        }

        var options = clientOptionsFactory.newBuilder(
                "",
                configOptional.get().getGrpcConfig()).build();
        List<HostAndPort> address = DiscoveryServices.resolve(configOptional.get().getGrpcConfig().getAddressesList());
        if (address.size() > 1) {
            throw new IllegalStateException("Expected only one address, but got: " + address.stream().map(Object::toString)
                    .collect(Collectors.joining(", ")));
        }
        GrpcTransport transport = new GrpcTransport(address.get(0),options);
        return new ProjectManagerAuthorizer(
                0,
                featureFlagsHolder,
                transport);
    }
}
