package ru.yandex.solomon.auth.authorizers;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.abc.AbcClient;
import ru.yandex.abc.dto.AbcMember;
import ru.yandex.abc.dto.Person;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthSubject;
import ru.yandex.solomon.auth.tvm.TvmSubject;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AbcAuthorizer implements Authorizer {
    private final AbcClient abcClient;

    public AbcAuthorizer(AbcClient abcClient) {
        this.abcClient = abcClient;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return subject instanceof OAuthSubject
                || subject instanceof SessionIdAuthSubject
                || subject instanceof AsUserSubject
                || subject instanceof TvmSubject
                || (subject instanceof OpenIdSubject && ((OpenIdSubject) subject).getLogin().isPresent());
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.ABC;
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject object, Permission permission) {
        String uid = RoleAuthorizer.getUid(subject);
        if (uid == null) {
            return failedFuture(new IllegalStateException("empty login in " + subject));
        }
        try {
            if (object instanceof AuthorizationObject.AbcAuthorizationObject authorizationObject) {
                if (!authorizationObject.roleScopeSlug().equals("administration")) {
                    return failedFuture(new IllegalArgumentException("unsupported abc role scope '" + authorizationObject.roleScopeSlug() + "'"));
                }
                if (authorizationObject.abcSlug().isBlank()) {
                    return failedFuture(new IllegalArgumentException("abc slug must be specified"));
                }
                return abcClient.getServiceMembers(authorizationObject.abcSlug(), authorizationObject.roleScopeSlug())
                        .thenApply(abcMembers -> {
                            var logins = abcMembers.stream()
                                    .map(AbcMember::person)
                                    .map(Person::login)
                                    .collect(Collectors.toSet());
                            if (logins.contains(uid)) {
                                return new Account(subject.getUniqueId(), subject.getAuthType(), AuthorizationType.ABC, RoleSet.PROJECT_ALL);
                            }
                            String message = "You(" + subject + ") have not " + permission.getSlug() + " permission \"" + authorizationObject.roleScopeSlug() + "\" in abc \"" + authorizationObject.abcSlug() + "\"";
                            throw new AuthorizationException(message);
                        });
            }
        } catch (Throwable t) {
            return failedFuture(t);
        }
        return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + object));
    }
}
