package ru.yandex.solomon.auth.authorizers;

import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationObjects;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class FeatureEnabledAuthorizer implements Authorizer {
    private final FeatureFlagsHolder featureFlagsHolder;
    private final Authorizer featureAuthorizer;
    private final Authorizer defaultAuthorizer;
    private final FeatureFlag projectFeatureFlag;

    public FeatureEnabledAuthorizer(
            Authorizer featureAuthorizer,
            Authorizer defaultAuthorizer,
            FeatureFlag projectFeatureFlag,
            FeatureFlagsHolder featureFlagsHolder)
    {
        this.featureAuthorizer = featureAuthorizer;
        this.defaultAuthorizer = defaultAuthorizer;
        this.projectFeatureFlag = projectFeatureFlag;
        this.featureFlagsHolder = featureFlagsHolder;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return defaultAuthorizer.canAuthorize(subject) || featureAuthorizer.canAuthorize(subject);
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return defaultAuthorizer.canAuthorizeObject(object) || featureAuthorizer.canAuthorizeObject(object);
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        return defaultAuthorizer.getAvailableAuthorizationObjects(subject, roles, types)
                .thenCompose(authorizationObjects -> {
                    return featureAuthorizer.getAvailableAuthorizationObjects(subject, roles, types)
                            .thenApply(featureObjects -> featureObjects.combine(authorizationObjects));
                });
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        if (authorizationObject.getType() == AuthorizationObject.Type.SERVICE_PROVIDER) {
            return featureAuthorizer.authorize(subject, authorizationObject, permission);
        }
        if (authorizationObject.getType() != AuthorizationObject.Type.CLASSIC) {
            return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + authorizationObject));
        }
        AuthorizationObject.ClassicAuthorizationObject object = (AuthorizationObject.ClassicAuthorizationObject) authorizationObject;

        boolean useFeatureAuth = featureFlagsHolder.hasFlag(projectFeatureFlag, object.projectId());
        return useFeatureAuth
                ? featureAuthorizer.authorize(subject, authorizationObject, permission)
                : defaultAuthorizer.authorize(subject, authorizationObject, permission);
    }
}
