package ru.yandex.solomon.auth.authorizers;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.NotFoundException;

/**
 * Special IAM authorizer for yc.yandex-team.ru: check only read permissions for project.
 * See https://st.yandex-team.ru/SOLOMON-7038 for details.
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class InternalIamAuthorizer implements Authorizer {
    private final SolomonConfHolder confHolder;

    public InternalIamAuthorizer(SolomonConfHolder confHolder) {
        this.confHolder = confHolder;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return subject.getAuthType() == AuthType.IAM;
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.CLASSIC;
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        if (authorizationObject.getType() != AuthorizationObject.Type.CLASSIC) {
            return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + authorizationObject));
        }
        AuthorizationObject.ClassicAuthorizationObject object = (AuthorizationObject.ClassicAuthorizationObject) authorizationObject;
        if (object.projectId().isEmpty()) {
            return CompletableFuture.failedFuture(new AuthorizationException("you(" + subject + ") have not " + permission.getSlug() + " permission in \"" + object.projectId()+"\""));
        }

        if (!RoleSet.PROJECT_VIEW.hasPermission(permission)) {
            return CompletableFuture.failedFuture(noPermissions(object.projectId(), permission, subject.toString()));
        }

        SolomonConfWithContext conf = confHolder.getConf();

        if (conf == null) {
            // TODO: it's not security to allow permissions here, please fix here: SOLOMON-7039
            return CompletableFuture.completedFuture(createAccount(subject));
        }

        Project project = conf.getProject(object.projectId());

        if (project == null) {
            return CompletableFuture.failedFuture(new NotFoundException("project " + object.projectId() + " not found"));
        }

        if (project.isOnlyAuthRead()) {
            return CompletableFuture.failedFuture(noPermissions(object.projectId(), permission, subject.toString()));
        }

        return CompletableFuture.completedFuture(createAccount(subject));
    }

    private static Account createAccount(AuthSubject subject) {
        return new Account(subject.getUniqueId(), subject.getAuthType(), AuthorizationType.IAM, RoleSet.PROJECT_VIEW);
    }

    private static AuthorizationException noPermissions(String projectId, Permission permission, String subject) {
        return new AuthorizationException("you(" + subject + ") have not permissions to " + permission.getSlug() + " in project " + projectId);
    }
}
