package ru.yandex.solomon.auth.authorizers;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthSubject;
import ru.yandex.solomon.auth.tvm.TvmSubject;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.NotOwnerException;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * TODO: will be replaced by IDM authorizer
 *
 * @author Sergey Polovko
 */
@Deprecated
@ParametersAreNonnullByDefault
public class ProjectAclAuthorizer implements Authorizer, AutoCloseable {
    protected static final String JUNK_PROJECT = "junk";

    private final SolomonConfHolder confHolder;

    public ProjectAclAuthorizer(SolomonConfHolder confHolder) {
        this.confHolder = confHolder;
    }

    @Override
    public void close() {
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.CLASSIC;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return subject instanceof OAuthSubject
            || subject instanceof SessionIdAuthSubject
            || subject instanceof AnonymousAuthSubject
            || subject instanceof AsUserSubject
            || subject instanceof TvmSubject
            || (subject instanceof OpenIdSubject && ((OpenIdSubject) subject).getLogin().isPresent());
    }

    @Override
    public CompletableFuture<Account> authorize(
        AuthSubject subject,
        AuthorizationObject object,
        Permission permission)
    {
        if (object.getType() != AuthorizationObject.Type.CLASSIC) {
            return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + object));
        }
        AuthorizationObject.ClassicAuthorizationObject authorizationObject = (AuthorizationObject.ClassicAuthorizationObject) object;
        try {
            String login = AuthSubject.getLogin(subject)
                    .orElseThrow(() -> new IllegalStateException("empty login in " + subject));

            Project project = getProjectById(authorizationObject.projectId());
            if (subject.getAuthType() == AuthType.Anonymous) {
                return authorizeForAnonymous(authorizationObject.projectId(), permission, project);
            }

            RoleSet roles = getRoles(login, authorizationObject.projectId(), project);
            if (roles.hasPermission(permission)) {
                return completedFuture(new Account(login, subject.getAuthType(), AuthorizationType.PROJECT_ACL, roles));
            }

            final String message;
            if (!authorizationObject.projectId().isEmpty()) {
                message = String.format("you(%s) have no permissions to %s in project '%s'", subject, permission.getSlug(), authorizationObject.projectId());
            } else if (!authorizationObject.folderId().isEmpty()) {
                message = String.format("you(%s) have no permissions to %s in folder '%s'", subject, permission.getSlug(), authorizationObject.folderId());
            } else {
                message = String.format("you(%s) have no permissions to %s in empty auth object", subject, permission.getSlug());
            }
            if (project != null) {
                return failedFuture(new NotOwnerException(message, project.getOwner()));
            }

            return failedFuture(new AuthorizationException(message));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Account> authorizeForAnonymous(
        String projectId,
        Permission permission,
        @Nullable Project project)
    {
        if (project != null && !project.isOnlyAuthRead()) {
            if (permission == Permission.DATA_READ
                || permission == Permission.METRICS_GET
                || permission == Permission.METRIC_NAMES_GET
                || permission == Permission.METRIC_LABELS_GET)
            {
                return CompletableFuture.completedFuture(Account.ANONYMOUS);
            }
        }
        String message = String.format("you(anonymous user) have no permissions to do anything in project '%s'", projectId);
        throw new AuthorizationException(message);
    }

    private static RoleSet getRoles(String login, String projectId, @Nullable Project project) {
        if (project == null) {
            if (StringUtils.isEmpty(projectId)) {
                // system wide roles
                return SolomonTeam.isMember(login) ? RoleSet.SYSTEM_ALL : RoleSet.EMPTY;
            }

            // unknown project, most probably it was just created and not yet cached
            // TODO: it's not security to allow permissions here, please fix here: SOLOMON-7039
            return RoleSet.PROJECT_ALL;
        }

        if (SolomonTeam.isMember(login)) {
            return RoleSet.ALL;
        }

        if (JUNK_PROJECT.equals(project.getId()) || login.equals(project.getOwner())) {
            // all users has full access to the 'junk' project and to their owning projects
            return RoleSet.PROJECT_ALL;
        }

        Acl acl = project.getAcl();
        boolean canRead = acl.getCanRead().contains(login);
        boolean canUpdate = acl.getCanUpdate().contains(login);
        boolean canDelete = acl.getCanDelete().contains(login);
        boolean canWrite = acl.getCanWrite().contains(login);

        if (canUpdate || canDelete) {
            return RoleSet.PROJECT_ALL;
        }

        if (canWrite) {
            return RoleSet.PROJECT_VIEW_AND_PUSH;
        }

        if (canRead || !project.isOnlyAuthRead()) {
            return RoleSet.PROJECT_VIEW;
        }

        return RoleSet.EMPTY;
    }

    @Nullable
    private Project getProjectById(String id) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return null;
        }
        return conf.getProject(id);
    }
}
