package ru.yandex.solomon.auth.authorizers;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationObjects;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.gateway.TProjectToCloudMappingConfig;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ProjectAclOrIamAuthorizer implements Authorizer {
    private final ProjectAclAuthorizer projectAclAuthorizer;
    private final Authorizer iamAuthorizer;
    private final Map<String, String> projectToCloudMapping;
    private final FeatureFlagsHolder featureFlagsHolder;

    public ProjectAclOrIamAuthorizer(
            ProjectAclAuthorizer projectAclAuthorizer,
            Authorizer iamAuthorizer,
            TGatewayCloudConfig gatewayCloudConfig,
            FeatureFlagsHolder featureFlagsHolder)
    {
        this.projectAclAuthorizer = projectAclAuthorizer;
        this.iamAuthorizer = iamAuthorizer;
        this.projectToCloudMapping = computeProjectToCloudMapping(gatewayCloudConfig);
        this.featureFlagsHolder = featureFlagsHolder;
    }

    private static Map<String, String> computeProjectToCloudMapping(TGatewayCloudConfig gatewayCloudConfig) {
        TProjectToCloudMappingConfig projectToCloudMappingConfig = gatewayCloudConfig.getProjectToCloudMapping();
        Map<String, String> projectToCloudMapping = new HashMap<>(projectToCloudMappingConfig.getMappingsCount());
        for (var mappingItem : projectToCloudMappingConfig.getMappingsList()) {
            projectToCloudMapping.put(mappingItem.getProjectId(), mappingItem.getCloudId());
        }
        return projectToCloudMapping;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return projectAclAuthorizer.canAuthorize(subject) || iamAuthorizer.canAuthorize(subject);
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return projectAclAuthorizer.canAuthorizeObject(object) || iamAuthorizer.canAuthorizeObject(object);
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        return projectAclAuthorizer.getAvailableAuthorizationObjects(subject, roles, types)
                .thenCompose(authorizationObjects -> {
                    return iamAuthorizer.getAvailableAuthorizationObjects(subject, roles, types)
                            .thenApply(featureObjects -> featureObjects.combine(authorizationObjects));
                });
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        if (authorizationObject instanceof AuthorizationObject.ClassicAuthorizationObject object) {
            return authorize(subject, object.projectId(), object.folderId(), permission);
        }
        if (authorizationObject instanceof AuthorizationObject.SpAuthorizationObject object) {
            if (StringUtils.isEmpty(object.objectId())) {
                return iamAuthorizer.authorize(subject, object, permission);
            }
            boolean useIamAuthorizerOnly = featureFlagsHolder.hasFlag(FeatureFlag.USE_IAM_AUTHORIZER_ONLY, object.objectId());
            if (useIamAuthorizerOnly) {
                return iamAuthorizer.authorize(subject, object, permission);
            } else {
                // todo (alextrushkin) remove after SOLOMON-7985
                return iamAuthorizer.authorize(subject, AuthorizationObject.serviceProvider(object.serviceProviderId(), null), permission);
            }
        }
        return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + authorizationObject));
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, String projectId, String folderId, Permission permission) {
        if (projectId.isEmpty()) {
            return authorizeUsingDefaultMethod(subject, projectId, folderId, permission);
        }

        String cloudId = projectToCloudMapping.getOrDefault(projectId, projectId);
        boolean useIamAuthorizerOnly = featureFlagsHolder.hasFlag(FeatureFlag.USE_IAM_AUTHORIZER_ONLY, projectId);

        if (useIamAuthorizerOnly) {
            return authorizeUsingIamOnly(subject, projectId, cloudId, folderId, permission);
        }

        return authorizeByProjectOrCloud(subject, projectId, cloudId, folderId, permission);
    }

    private CompletableFuture<Account> authorizeUsingIamOnly(
            AuthSubject subject,
            String projectId,
            String cloudId,
            String folderId,
            Permission permission)
    {
        if (iamAuthorizer.canAuthorize(subject)) {
            return iamAuthorizer.authorize(subject, cloudId, folderId, permission);
        }
        return CompletableFuture.failedFuture(new AuthorizationException("cannot authorize, please check that IAM authorization is enabled for " + projectId + " and you are logged in cloud console"));
    }

    private CompletableFuture<Account> authorizeUsingDefaultMethod(
            AuthSubject subject,
            String projectId,
            String folderId,
            Permission permission) {
        return projectAclAuthorizer.canAuthorize(subject)
                ? projectAclAuthorizer.authorize(subject, projectId, folderId, permission)
                : iamAuthorizer.authorize(subject, projectId, folderId, permission);
    }

    private CompletableFuture<Account> authorizeByProjectOrCloud(
            AuthSubject subject,
            String projectId,
            String cloudId,
            String folderId,
            Permission permission) {
        return projectAclAuthorizer.canAuthorize(subject)
                ? projectAclAuthorizer.authorize(subject, projectId, folderId, permission)
                : iamAuthorizer.authorize(subject, cloudId, folderId, permission);
    }
}
