package ru.yandex.solomon.auth.authorizers;

import java.time.Duration;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;

import com.google.common.base.Throwables;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.AccessServiceGrpc;
import ru.yandex.monitoring.api.v3.project.manager.AuthorizeRequest;
import ru.yandex.monitoring.api.v3.project.manager.AvailableResourcesRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationObjects;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.dto.AccessDtoConverter;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.util.async.InFlightLimiter;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Alexey Trushkin
 */
public class ProjectManagerAuthorizer implements Authorizer {
    private static final Logger logger = LoggerFactory.getLogger(ProjectManagerAuthorizer.class);
    private static final Duration CONNECTION_TIMEOUT = Duration.ofSeconds(5);
    private static final Duration REQUEST_TIMEOUT = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(ProjectManagerAuthorizer::needToRetry)
            .withNumRetries(5)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final InFlightLimiter limiter;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final GrpcTransport transport;

    public ProjectManagerAuthorizer(
            int maxInflight,
            FeatureFlagsHolder featureFlagsHolder,
            GrpcTransport transport)
    {
        this.transport = transport;
        this.limiter = new InFlightLimiter(maxInflight != 0 ? maxInflight : 500);
        this.featureFlagsHolder = featureFlagsHolder;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_AUTHORIZER, "ProjectManagerAuthorizer");
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.CLASSIC
                || object.getType() == AuthorizationObject.Type.SERVICE_PROVIDER
                || object.getType() == AuthorizationObject.Type.ABC;
    }

    @Override
    public CompletableFuture<Account> authorize(
            AuthSubject subject,
            AuthorizationObject authorizationObject,
            Permission permission) {
        return runWithRetries(() -> auth(subject, authorizationObject, permission), RETRY_CONFIG);
    }

    private CompletableFuture<Account> auth(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        try {
            var request = AuthorizeRequest.newBuilder()
                    .setSubject(AccessDtoConverter.toProto(subject))
                    .setPermission(permission.getSlug())
                    .setResourcePath(AccessDtoConverter.toProto(authorizationObject))
                    .build();
            var result = new CompletableFuture<Account>();
            limiter.run(() -> transport.unaryCall(AccessServiceGrpc.getAuthorizeMethod(), request)
                    .whenComplete((response, t) -> {
                try {
                    if (t != null) {
                        result.completeExceptionally(handleException(t));
                    } else {
                        result.complete(new Account(subject.getUniqueId(),
                                subject.getAuthType(),
                                AccessDtoConverter.fromProto(response.getAuthorizationType()),
                                RoleSet.of(AccessDtoConverter.fromProto(response.getRolesList()))));
                    }
                } catch (Throwable x) {
                    result.completeExceptionally(x);
                }
            }));
            return result;
        } catch (Throwable x) {
            return CompletableFuture.failedFuture(x);
        }
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        if (!featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_AUTHORIZER, "ProjectManagerAuthorizer.getAvailableAuthorizationObjects")) {
            return CompletableFuture.completedFuture(AuthorizationObjects.EMPTY);
        }
        try {
            var request = AvailableResourcesRequest.newBuilder()
                    .setSubject(AccessDtoConverter.toProto(subject))
                    .addAllRoles(AccessDtoConverter.toProto(roles))
                    .build();
            var result = new CompletableFuture<AuthorizationObjects>();
            limiter.run(() -> transport.unaryCall(AccessServiceGrpc.getAvailableResourcesMethod(), request)
                    .whenComplete((response, t) -> {
                try {
                    if (t != null) {
                        result.completeExceptionally(handleException(t));
                    } else {
                        result.complete(AccessDtoConverter.fromProtoResourcePath(response.getResourcePathList()));
                    }
                } catch (Throwable x) {
                    result.completeExceptionally(x);
                }
            }));
            return result;
        } catch (Throwable x) {
            return CompletableFuture.failedFuture(x);
        }
    }

    private static boolean needToRetry(Throwable throwable) {
        Status status = Status.fromThrowable(throwable);
        return status.getCode() == Status.Code.UNAVAILABLE;
    }

    private Exception handleException(@Nonnull Throwable throwable) {
        throwable = CompletableFutures.unwrapCompletionException(throwable);
        Throwables.throwIfUnchecked(throwable);
        return new AuthorizationException(throwable.getMessage(), throwable);
    }

}
