package ru.yandex.solomon.auth.authorizers;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.GroupMember;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationObjects;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthSubject;
import ru.yandex.solomon.auth.tvm.TvmSubject;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.core.exceptions.NotOwnerException;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.getNow;
import static ru.yandex.solomon.auth.authorizers.ProjectAclAuthorizer.JUNK_PROJECT;

/**
 * @author Alexey Trushkin
 */
public class RoleAuthorizer implements Authorizer, AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(RoleAuthorizer.class);
    private static final long UPDATER_START_DELAY_MILLIS = 1000;
    private static final Duration UPDATE_INTERVAL = Duration.ofMinutes(5);

    private final SolomonConfHolder confHolder;
    private final ProjectsDao projectsDao;
    private final ServiceProvidersDao serviceProvidersDao;
    private final StateUpdater stateUpdater;

    private volatile RolesState rolesState = RolesState.EMPTY;

    public RoleAuthorizer(
            SolomonConfHolder confHolder,
            MetricRegistry registry,
            ThreadPoolProvider threads,
            ProjectAclEntryDao projectAclEntryDao,
            SystemAclEntryDao systemAclEntryDao,
            ServiceProviderAclEntryDao serviceProviderAclEntryDao,
            ServiceProvidersDao serviceProvidersDao,
            GroupMemberDao groupMemberDao,
            ProjectsDao projectsDao)
    {
        this.confHolder = confHolder;
        this.serviceProvidersDao = serviceProvidersDao;
        this.projectsDao = projectsDao;
        stateUpdater = new StateUpdater(threads, registry, projectAclEntryDao, systemAclEntryDao, serviceProviderAclEntryDao, groupMemberDao);
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return subject instanceof OAuthSubject
                || subject instanceof SessionIdAuthSubject
                || subject instanceof AnonymousAuthSubject
                || subject instanceof AsUserSubject
                || subject instanceof TvmSubject
                || (subject instanceof OpenIdSubject && ((OpenIdSubject) subject).getLogin().isPresent());
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.CLASSIC || object.getType() == AuthorizationObject.Type.SERVICE_PROVIDER;
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        String uid = getUid(subject);
        if (uid == null) {
            return failedFuture(new IllegalStateException("empty login in " + subject));
        }
        if (!canAuthorize(subject) || types.isEmpty()) {
            return CompletableFuture.completedFuture(AuthorizationObjects.EMPTY);
        }
        AclUidType type = getAclUidType(subject);
        var objects = Stream.<AuthorizationObject>concat(
                types.contains(AuthorizationObject.Type.CLASSIC)
                        ? getAvailableProjects(uid, type, roles).stream().map(s -> AuthorizationObject.classic(s, ""))
                        : Stream.of(),
                types.contains(AuthorizationObject.Type.SERVICE_PROVIDER)
                        ? getAvailableServiceProviders(uid, type, roles).stream().map(s -> AuthorizationObject.serviceProvider(s, null))
                        : Stream.of()
        ).collect(Collectors.toSet());

        return CompletableFuture.completedFuture(AuthorizationObjects.of(objects));
    }

    private Set<String> getAvailableServiceProviders(String uid, AclUidType type, Set<Role> roles) {
        if (isSystemLogin(uid, type)) {
            return rolesState.idToServiceProviderEntry.keySet().stream()
                    .map(ServiceProviderAclEntry.Id::serviceProviderId)
                    .collect(Collectors.toSet());
        }
        var userSps = rolesState.getServiceProvidersBySubject(uid, type);
        var userGroupSps = rolesState.getServiceProvidersBySubjectGroup(uid);
        return Stream.concat(userSps.stream(), userGroupSps.stream())
                .filter(objectRoles -> {
                    for (Role role : roles) {
                        if (objectRoles.roles.contains(role.name())) {
                            return true;
                        }
                    }
                    return false;
                })
                .map(ObjectRoles::id)
                .collect(Collectors.toSet());
    }

    private Set<String> getAvailableProjects(String uid, AclUidType type, Set<Role> roles) {
        if (isSystemLogin(uid, type)) {
            return rolesState.idToProjectEntry.keySet().stream()
                    .map(ProjectAclEntry.Id::projectId)
                    .collect(Collectors.toSet());
        }
        var userProjects = rolesState.getProjectsBySubject(uid, type);
        var userGroupProjects = rolesState.getProjectsBySubjectGroup(uid);
        return Stream.concat(userProjects.stream(), userGroupProjects.stream())
                .filter(projectRoles -> {
                    for (Role role : roles) {
                        if (projectRoles.roles.contains(role.name())) {
                            return true;
                        }
                    }
                    return false;
                })
                .map(ObjectRoles::id)
                .collect(Collectors.toSet());
    }

    @Override
    public CompletableFuture<Account> authorize(
            AuthSubject subject,
            AuthorizationObject object,
            Permission permission)
    {
        String uid = getUid(subject);
        if (uid == null) {
            return failedFuture(new IllegalStateException("empty login in " + subject));
        }
        try {
            if (object instanceof AuthorizationObject.ClassicAuthorizationObject authorizationObject) {
                return getProjectById(authorizationObject.projectId())
                        .thenCompose(
                            projectOptional -> tryAuthorize(
                                subject,
                                authorizationObject.projectId(),
                                authorizationObject.folderId(),
                                permission,
                                uid,
                                projectOptional));

            } else if (object instanceof AuthorizationObject.SpAuthorizationObject authorizationObject) {
                return getServiceProviderById(authorizationObject.serviceProviderId())
                        .thenCompose(spOptional -> tryAuthorize(subject, authorizationObject.serviceProviderId(), permission, uid, spOptional));
            }
        } catch (Throwable t) {
            return failedFuture(t);
        }
        return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + object));
    }

    protected static String getUid(AuthSubject subject) {
        var loginOptional = AuthSubject.getLogin(subject);
        if (loginOptional.isEmpty()) {
            return null;
        }
        if (subject instanceof TvmSubject.ServiceSubject serviceSubject) {
            //fix tvm- prefix
            return Integer.toString(serviceSubject.getClientId());
        } else {
            return loginOptional.get();
        }
    }

    private CompletableFuture<Account> tryAuthorize(
            AuthSubject subject,
            String projectId,
            String folderId,
            Permission permission,
            String login,
            Optional<Project> projectOptional)
    {
        AclUidType type = getAclUidType(subject);
        RoleSet roleSet;
        if (projectId.isEmpty()  || projectOptional.isEmpty()) {
            // system wide roles
            if (isSystemLogin(login, type)) {
                roleSet = RoleSet.SYSTEM_ALL;
            } else {
                return failedFuture(new NotFoundException(String.format("no project with id '%s'", projectId)));
            }
        } else {
            roleSet = getRoles(login, type, projectOptional.get());
        }
        if (subject.getAuthType() == AuthType.Anonymous) {
            return authorizeForAnonymous(projectId, permission, projectOptional.get());
        }
        if (roleSet.hasPermission(permission)) {
            return completedFuture(new Account(login, subject.getAuthType(), AuthorizationType.ROLE, roleSet));
        }
        //errors
        String objStr = projectId.isEmpty() ? folderId : projectId;
        String message = "You(" + subject + ") have not " + permission.getSlug() + " permission in \"" + objStr + "\"";
        if (projectId.isEmpty() || folderId.isEmpty()) {
            return failedFuture(new AuthorizationException(message));
        }
        if (projectOptional.isPresent()) {
            return failedFuture(new NotOwnerException(message, projectOptional.get().getOwner()));
        }
        return failedFuture(new AuthorizationException(message));
    }

    private RoleSet getRoles(String login, AclUidType type, Project project) {
        if (isSystemLogin(login, type)) {
            return RoleSet.ALL;
        }

        if (JUNK_PROJECT.equals(project.getId())) {
            // all users has full access to the 'junk' project and to their owning projects
            return RoleSet.PROJECT_ALL;
        }

        //login project roles
        ProjectAclEntry loginEntry = rolesState.getProjectAclEntry(ProjectAclEntry.compositeId(project.getId(), login, type));
        //login system roles
        SystemAclEntry loginSystemEntry = rolesState.getSystemAclEntry(SystemAclEntry.compositeId(login, type));
        //login groups project roles
        List<ProjectAclEntry> loginGroupEntries = rolesState.getProjectAclEntryByLoginGroups(login, project.getId());
        //login groups system roles
        List<SystemAclEntry> loginSystemGroupEntries = rolesState.getSystemAclEntryByLoginGroups(login);
        //concat all roles
        List<Role> collect = Stream.concat(loginGroupEntries.stream(), Stream.of(loginEntry))
                .filter(Objects::nonNull)
                .flatMap(projectAclEntry -> projectAclEntry.getRoles().stream())
                .distinct()
                .map(Role::fromString)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        Stream.concat(loginSystemGroupEntries.stream(), Stream.of(loginSystemEntry))
                .filter(Objects::nonNull)
                .flatMap(aclEntry -> aclEntry.getRoles().stream())
                .distinct()
                .map(Role::fromString)
                .filter(Objects::nonNull)
                .forEach(collect::add);

        if (login.equals(project.getOwner())) {
            // all users has full access to their owning projects
            collect.addAll(Arrays.asList(Role.PROJECT_ROLES));
        }
        if (!project.isOnlyAuthRead()) {
            collect.add(Role.VIEWER);
        }
        return RoleSet.of(collect);
    }

    private CompletableFuture<Account> authorizeForAnonymous(
            String projectId,
            Permission permission,
            @Nullable Project project)
    {
        if (project != null && !project.isOnlyAuthRead()) {
            if (permission == Permission.DATA_READ
                    || permission == Permission.METRICS_GET
                    || permission == Permission.METRIC_NAMES_GET
                    || permission == Permission.METRIC_LABELS_GET)
            {
                return CompletableFuture.completedFuture(Account.ANONYMOUS);
            }
        }
        String message = String.format("you(anonymous user) have no permissions to do anything in project '%s'", projectId);
        throw new AuthorizationException(message);
    }

    private CompletableFuture<Account> tryAuthorize(
            AuthSubject subject,
            String objectId,
            Permission permission,
            String login,
            Optional<ServiceProvider> spOptional)
    {
        AclUidType type = getAclUidType(subject);
        RoleSet roleSet;
        if (StringUtils.isEmpty(objectId) || spOptional.isEmpty()) {
            // system wide roles
            roleSet = isSystemLogin(login, type) ? RoleSet.SYSTEM_ALL : RoleSet.EMPTY;
        } else {
            roleSet = getRoles(login, type, spOptional.get());
        }
        if (roleSet.hasPermission(permission)) {
            return completedFuture(new Account(login, subject.getAuthType(), AuthorizationType.ROLE, roleSet));
        }
        //errors
        String message = "You(" + subject + ") have not " + permission.getSlug() + " permission in service provider \"" + objectId + "\"";
        return failedFuture(new AuthorizationException(message));
    }

    private RoleSet getRoles(String login, AclUidType type, ServiceProvider sp) {
        if (isSystemLogin(login, type)) {
            return RoleSet.ALL;
        }
        //login sp roles
        ServiceProviderAclEntry loginEntry = rolesState.getServiceProviderAclEntry(ServiceProviderAclEntry.compositeId(sp.getId(), login, type));
        //login groups project roles
        List<ServiceProviderAclEntry> loginGroupEntries = rolesState.getServiceProviderAclEntryByLoginGroups(login, sp.getId());
        //concat all roles
        List<Role> collect = Stream.concat(loginGroupEntries.stream(), Stream.of(loginEntry))
                .filter(Objects::nonNull)
                .flatMap(entry -> entry.getRoles().stream())
                .distinct()
                .map(Role::fromString)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return RoleSet.of(collect);
    }

    private AclUidType getAclUidType(AuthSubject subject) {
        return subject.getAuthType() == AuthType.TvmService ? AclUidType.TVM : AclUidType.USER;
    }

    private boolean isSystemLogin(String login, AclUidType type) {
        return SolomonTeam.isMember(login) || rolesState.isSystemLogin(login, type);
    }

    private CompletableFuture<Optional<Project>> getProjectById(String id) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return projectsDao.findById(id);
        }
        var project = conf.getProject(id);
        if (project == null) {
            return projectsDao.findById(id);
        }
        return CompletableFuture.completedFuture(Optional.ofNullable(project));
    }

    private CompletableFuture<Optional<ServiceProvider>> getServiceProviderById(String id) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return serviceProvidersDao.read(id);
        }
        var sp = conf.getServiceProvider(id);
        if (sp == null) {
            return serviceProvidersDao.read(id);
        }
        return CompletableFuture.completedFuture(Optional.ofNullable(sp));
    }

    @Override
    public void close() throws Exception {
        stateUpdater.close();
    }

    private class StateUpdater implements AutoCloseable {
        private final PingActorRunner actor;
        private final AsyncMetrics metrics;
        private final ProjectAclEntryDao projectAclEntryDao;
        private final SystemAclEntryDao systemAclEntryDao;
        private final ServiceProviderAclEntryDao serviceProviderAclEntryDao;
        private final GroupMemberDao groupMemberDao;
        private volatile boolean closed;

        private StateUpdater(
                ThreadPoolProvider threads,
                MetricRegistry registry,
                ProjectAclEntryDao projectAclEntryDao,
                SystemAclEntryDao systemAclEntryDao,
                ServiceProviderAclEntryDao serviceProviderAclEntryDao,
                GroupMemberDao groupMemberDao)
        {
            this.projectAclEntryDao = projectAclEntryDao;
            this.systemAclEntryDao = systemAclEntryDao;
            this.serviceProviderAclEntryDao = serviceProviderAclEntryDao;
            this.groupMemberDao = groupMemberDao;
            this.metrics = new AsyncMetrics(registry, "authorizer.role.update");
            this.actor = PingActorRunner.newBuilder()
                    .executor(threads.getExecutorService("CpuLowPriority", ""))
                    .timer(threads.getSchedulerExecutorService())
                    .operation("Update role authorizer state")
                    .pingInterval(UPDATE_INTERVAL)
                    .backoffDelay(Duration.ofMinutes(1))
                    .onPing(this::updateState)
                    .build();
            long delay = DurationUtils.randomize(UPDATER_START_DELAY_MILLIS);
            threads.getSchedulerExecutorService().schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
        }

        private CompletableFuture<?> updateState(int attempts) {
            if (closed) {
                return completedFuture(null);
            }
            var future = loadData(projectAclEntryDao.getAll(),
                    systemAclEntryDao.getAll(), groupMemberDao.getAll(), serviceProviderAclEntryDao.getAll())
                    .thenApply(this::parseRolesState)
                    .whenComplete((result, throwable) -> {
                        if (throwable != null) {
                            logger.error("Cannot update role authorizer state", throwable);
                        } else {
                            rolesState = result;
                            logger.debug("Role authorizer state updated");
                        }
                    });
            metrics.forFuture(future);
            return future;
        }

        private RolesState parseRolesState(Cortege cortege) {
            var projectsEntries = cortege.projectAclEntries.stream()
                    .collect(Collectors.toMap(ProjectAclEntry::getCompositeId, Function.identity()));
            var systemEntries = cortege.systemAclEntries.stream()
                    .collect(Collectors.toMap(SystemAclEntry::getCompositeId, Function.identity()));
            var serviceProvidersEntries = cortege.providerAclEntries.stream()
                    .collect(Collectors.toMap(ServiceProviderAclEntry::getCompositeId, Function.identity()));
            Map<String, List<String>> userIdToGroupId = new HashMap<>();
            for (var groupMember : cortege.groupMembers) {
                List<String> groups = userIdToGroupId.getOrDefault(groupMember.userId(), new ArrayList<>());
                groups.add(groupMember.groupId());
                userIdToGroupId.put(groupMember.userId(), groups);
            }
            Map<SubjectId, List<ObjectRoles>> subjectToProjects = new HashMap<>();
            for (var projectAclEntry : cortege.projectAclEntries) {
                var subject = new SubjectId(projectAclEntry.getUid(), projectAclEntry.getType());
                subjectToProjects.computeIfAbsent(subject, subjectId -> new ArrayList<>())
                        .add(new ObjectRoles(projectAclEntry.getProjectId(), projectAclEntry.getRoles()));
            }
            Map<SubjectId, List<ObjectRoles>> subjectIdsToServiceProviders = new HashMap<>();
            for (var spAclEntry : cortege.providerAclEntries) {
                var subject = new SubjectId(spAclEntry.getUid(), spAclEntry.getType());
                subjectIdsToServiceProviders.computeIfAbsent(subject, subjectId -> new ArrayList<>())
                        .add(new ObjectRoles(spAclEntry.getServiceProviderId(), spAclEntry.getRoles()));
            }
            return new RolesState(systemEntries, projectsEntries, serviceProvidersEntries, userIdToGroupId, subjectToProjects, subjectIdsToServiceProviders);
        }

        private CompletableFuture<Cortege> loadData(
                CompletableFuture<List<ProjectAclEntry>> a,
                CompletableFuture<List<SystemAclEntry>> b,
                CompletableFuture<List<GroupMember>> c,
                CompletableFuture<List<ServiceProviderAclEntry>> d)
        {
            CompletableFuture<?>[] futuresArray = new CompletableFuture[]{a, b, c, d};
            CompletableFuture<Void> allOfFuture = CompletableFuture.allOf(futuresArray);
            return allOfFuture.thenApply(v -> new Cortege(getNow(a), getNow(b), getNow(c), getNow(d)));
        }

        @Override
        public void close() {
            closed = true;
            actor.close();
        }

    }

    @VisibleForTesting
    public static class RolesState {
        private static final RolesState EMPTY = new RolesState();
        private final Map<SystemAclEntry.Id, SystemAclEntry> uidToSystemEntry;
        private final Map<ProjectAclEntry.Id, ProjectAclEntry> idToProjectEntry;
        private final Map<ServiceProviderAclEntry.Id, ServiceProviderAclEntry> idToServiceProviderEntry;
        private final Map<SubjectId, List<ObjectRoles>> subjectIdsToProjects;
        private final Map<SubjectId, List<ObjectRoles>> subjectIdsToServiceProviders;
        private final Map<String, List<String>> userIdToGroupId;

        private RolesState() {
            uidToSystemEntry = Collections.emptyMap();
            idToProjectEntry = Collections.emptyMap();
            userIdToGroupId = Collections.emptyMap();
            idToServiceProviderEntry = Collections.emptyMap();
            subjectIdsToProjects = Collections.emptyMap();
            subjectIdsToServiceProviders = Collections.emptyMap();
        }

        @VisibleForTesting
        public RolesState(
                Map<SystemAclEntry.Id, SystemAclEntry> uidToSystemEntry,
                Map<ProjectAclEntry.Id, ProjectAclEntry> idToProjectEntry,
                Map<ServiceProviderAclEntry.Id, ServiceProviderAclEntry> idToServiceProviderEntry,
                Map<String, List<String>> userIdToGroupId,
                Map<SubjectId, List<ObjectRoles>> subjectIdsToProjects,
                Map<SubjectId, List<ObjectRoles>> subjectIdsToServiceProviders)
        {
            this.uidToSystemEntry = Objects.requireNonNull(uidToSystemEntry);
            this.idToProjectEntry = Objects.requireNonNull(idToProjectEntry);
            this.userIdToGroupId = Objects.requireNonNull(userIdToGroupId);
            this.idToServiceProviderEntry = Objects.requireNonNull(idToServiceProviderEntry);
            this.subjectIdsToProjects = Objects.requireNonNull(subjectIdsToProjects);
            this.subjectIdsToServiceProviders = Objects.requireNonNull(subjectIdsToServiceProviders);
        }

        public ProjectAclEntry getProjectAclEntry(ProjectAclEntry.Id id) {
            return idToProjectEntry.get(id);
        }

        public List<ProjectAclEntry> getProjectAclEntryByLoginGroups(String login, String projectId) {
            return userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(groupId -> idToProjectEntry.get(ProjectAclEntry.compositeId(projectId, groupId, AclUidType.GROUP)))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
        }

        public SystemAclEntry getSystemAclEntry(SystemAclEntry.Id id) {
            return uidToSystemEntry.get(id);
        }

        public List<SystemAclEntry> getSystemAclEntryByLoginGroups(String login) {
            return userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(groupId -> uidToSystemEntry.get(SystemAclEntry.compositeId(groupId, AclUidType.GROUP)))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
        }

        public ServiceProviderAclEntry getServiceProviderAclEntry(ServiceProviderAclEntry.Id id) {
            return idToServiceProviderEntry.get(id);
        }

        public List<ServiceProviderAclEntry> getServiceProviderAclEntryByLoginGroups(String login, String serviceProvider) {
            return userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(groupId -> idToServiceProviderEntry.get(ServiceProviderAclEntry.compositeId(serviceProvider, groupId, AclUidType.GROUP)))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
        }

        public boolean isSystemLogin(String login, AclUidType type) {
            //login admin
            var systemLogin = uidToSystemEntry.get(SystemAclEntry.compositeId(login, type));
            if (systemLogin != null) {
                if (systemLogin.getRoles().contains(Role.ADMIN.name())) {
                    return true;
                }
            }
            //login in admin group
            List<SystemAclEntry.Id> groupIds = userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(s -> SystemAclEntry.compositeId(s, AclUidType.GROUP))
                    .collect(Collectors.toList());
            for (SystemAclEntry.Id groupId : groupIds) {
                var systemGroup = uidToSystemEntry.get(groupId);
                if (systemGroup == null) {
                    continue;
                }
                if (systemGroup.getRoles().contains(Role.ADMIN.name())) {
                    return true;
                }
            }
            return false;
        }

        public List<ObjectRoles> getProjectsBySubject(String login, AclUidType type) {
            return subjectIdsToProjects.getOrDefault(new SubjectId(login, type), List.of());
        }

        public List<ObjectRoles> getProjectsBySubjectGroup(String login) {
            return userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(groupId -> subjectIdsToProjects.get(new SubjectId(groupId, AclUidType.GROUP)))
                    .filter(Objects::nonNull)
                    .flatMap(Collection::stream)
                    .collect(Collectors.toList());
        }

        public List<ObjectRoles> getServiceProvidersBySubject(String login, AclUidType type) {
            return subjectIdsToServiceProviders.getOrDefault(new SubjectId(login, type), List.of());
        }

        public List<ObjectRoles> getServiceProvidersBySubjectGroup(String login) {
            return userIdToGroupId.getOrDefault(login, Collections.emptyList()).stream()
                    .map(groupId -> subjectIdsToServiceProviders.get(new SubjectId(groupId, AclUidType.GROUP)))
                    .filter(Objects::nonNull)
                    .flatMap(Collection::stream)
                    .collect(Collectors.toList());
        }
    }

    private static record Cortege(List<ProjectAclEntry> projectAclEntries, List<SystemAclEntry> systemAclEntries,
                                  List<GroupMember> groupMembers, List<ServiceProviderAclEntry> providerAclEntries) {
    }

    public static record SubjectId(String uid, AclUidType type) {
    }

    public static record ObjectRoles(String id, Set<String> roles) {
    }
}
