package ru.yandex.solomon.auth.authorizers;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.AuthorizerDelegate;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.ServiceProvider;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Alexey Trushkin
 */
public class ServiceProviderAuthorizer extends AuthorizerDelegate {
    private final SolomonConfHolder confHolder;
    private final ServiceProvidersDao serviceProvidersDao;

    public ServiceProviderAuthorizer(
            Authorizer delegate,
            SolomonConfHolder confHolder,
            ServiceProvidersDao serviceProvidersDao)
    {
        super(delegate);
        this.confHolder = confHolder;
        this.serviceProvidersDao = serviceProvidersDao;
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        if (authorizationObject instanceof AuthorizationObject.SpAuthorizationObject spAuthorizationObject) {
            return getServiceProviderById(spAuthorizationObject.serviceProviderId())
                    .thenCompose(serviceProviderOptional -> {
                        // its Service provider auth
                        if (serviceProviderOptional.isEmpty()) {
                            String message = "You(" + subject + ") have not " + permission.getSlug() + " permission in service provider \"" + spAuthorizationObject.serviceProviderId() + "\"";
                            return failedFuture(new AuthorizationException(message));
                        }
                        if (!serviceProviderOptional.get().getIamServiceAccountIds().contains(subject.getUniqueId())) {
                            return failedFuture(new AuthorizationException("account id differs from the provider's one"));
                        }
                        return super.authorize(subject, authorizationObject, permission);
                    });
        }
        return super.authorize(subject, authorizationObject, permission);
    }

    private CompletableFuture<Optional<ServiceProvider>> getServiceProviderById(String id) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return serviceProvidersDao.read(id);
        }
        var sp = conf.getServiceProvider(id);
        if (sp == null) {
            return serviceProvidersDao.read(id);
        }
        return CompletableFuture.completedFuture(Optional.of(sp));
    }

}
