package ru.yandex.solomon.auth.dto;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.api.v3.project.manager.AuthorizeResponse;
import ru.yandex.monitoring.api.v3.project.manager.AvailableResourcesResponse;
import ru.yandex.monitoring.api.v3.project.manager.Resource;
import ru.yandex.monitoring.api.v3.project.manager.ResourcePath;
import ru.yandex.monitoring.api.v3.project.manager.Role;
import ru.yandex.monitoring.api.v3.project.manager.Subject;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationObjects;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.iam.IamSubject;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthSubject;
import ru.yandex.solomon.auth.tvm.TvmSubject;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AccessDtoConverter {
    public static final String PROJECT_TYPE = "project-manager.project";
    public static final String ABC_TYPE = "project-manager.abc";
    public static final String ABC_ROLE_TYPE = "project-manager.abc.role";
    public static final String CLOUD_TYPE = "project-manager.cloud";
    public static final String FOLDER_TYPE = "project-manager.folder";
    public static final String SERVICE_PROVIDER_TYPE = "project-manager.service-provider";

    public static AuthSubject fromProto(Subject callSubject) {
        validate(callSubject);
        switch (callSubject.getTypeCase()) {
            case IAM_SERVICE_SUBJECT -> {
                return new IamSubject(new IamSubject.SimpleSubject(new yandex.cloud.auth.api.Subject.ServiceAccount.Id(callSubject.getIamServiceSubject().getId())));
            }
            case IAM_USER_SUBJECT -> {
                return new IamSubject(new IamSubject.SimpleSubject(new yandex.cloud.auth.api.Subject.UserAccount.Id(callSubject.getIamUserSubject().getId())));
            }
            case TVM_USER_SUBJECT -> {
                return TvmSubject.user(callSubject.getTvmUserSubject().getUid(), callSubject.getTvmUserSubject().getLogin());
            }
            case TVM_SERVICE_SUBJECT -> {
                return TvmSubject.service(callSubject.getTvmServiceSubject().getClientId());
            }
            case ANONYMOUS_ACCOUNT -> {
                return AnonymousAuthSubject.INSTANCE;
            }
            case O_AUTH_SUBJECT -> {
                return new OAuthSubject(callSubject.getOAuthSubject().getUid(), callSubject.getOAuthSubject().getLogin());
            }
            case OPEN_ID_SUBJECT -> {
                return new OpenIdSubject(callSubject.getOpenIdSubject().getAccountId(), Optional.of(callSubject.getOpenIdSubject().getLogin()));
            }
            case SESSION_ID_COOKIE_SUBJECT -> {
                return new SessionIdAuthSubject(callSubject.getSessionIdCookieSubject().getLogin());
            }
            default -> throw new UnsupportedOperationException(callSubject + " not supported");
        }
    }

    private static void validate(Subject callSubject) {
        switch (callSubject.getTypeCase()) {
            case IAM_SERVICE_SUBJECT -> {
                if (callSubject.getIamServiceSubject().getId().isEmpty()) {
                    throw new BadRequestException("subject Id can't be blank");
                }
            }
            case IAM_USER_SUBJECT -> {
                if (callSubject.getIamUserSubject().getId().isEmpty()) {
                    throw new BadRequestException("subject Id can't be blank");
                }
            }
            case TVM_USER_SUBJECT -> {
                if (callSubject.getTvmUserSubject().getLogin().isEmpty()) {
                    throw new BadRequestException("subject login can't be blank");
                }
            }
            case O_AUTH_SUBJECT -> {
                if (callSubject.getOAuthSubject().getLogin().isEmpty()) {
                    throw new BadRequestException("subject Id can't be blank");
                }
            }
            case OPEN_ID_SUBJECT -> {
                if (callSubject.getOpenIdSubject().getAccountId().isEmpty()) {
                    throw new BadRequestException("subject account Id can't be blank");
                }
            }
            case SESSION_ID_COOKIE_SUBJECT -> {
                if (callSubject.getSessionIdCookieSubject().getLogin().isEmpty()) {
                    throw new BadRequestException("subject login can't be blank");
                }
            }
        }
    }

    public static Set<ru.yandex.solomon.auth.roles.Role> fromProto(List<Role> rolesList) {
        Set<ru.yandex.solomon.auth.roles.Role> roles = new HashSet<>();
        for (Role role : rolesList) {
            switch (role) {
                case ROLE_VIEWER -> roles.add(ru.yandex.solomon.auth.roles.Role.VIEWER);
                case ROLE_PROJECT_ADMIN -> roles.add(ru.yandex.solomon.auth.roles.Role.PROJECT_ADMIN);
                case ROLE_JNS_SENDER -> roles.add(ru.yandex.solomon.auth.roles.Role.JNS_SENDER);
                case ROLE_EDITOR -> roles.add(ru.yandex.solomon.auth.roles.Role.EDITOR);
                case ROLE_PUSHER -> roles.add(ru.yandex.solomon.auth.roles.Role.PUSHER);
                case ROLE_SERVICE_PROVIDER_ADMIN -> roles.add(ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ADMIN);
                case ROLE_SERVICE_PROVIDER_ALERT_EDITOR -> roles.add(ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ALERT_EDITOR);
                case ROLE_BETA_TESTER -> roles.add(ru.yandex.solomon.auth.roles.Role.BETA_TESTER);
                case ROLE_MUTER -> roles.add(ru.yandex.solomon.auth.roles.Role.MUTER);
                case ROLE_MONITORING_ADMIN -> roles.add(ru.yandex.solomon.auth.roles.Role.ADMIN);
                case ROLE_MONITORING_PROJECT_DELETER -> roles.add(ru.yandex.solomon.auth.roles.Role.PROJECT_DELETER);
                default -> throw new UnsupportedOperationException("Unsupported role: " + role);
            }
        }
        return roles;
    }

    public static Set<Role> toProto(Set<ru.yandex.solomon.auth.roles.Role> rolesList) {
        Set<Role> roles = new HashSet<>();
        for (ru.yandex.solomon.auth.roles.Role role : rolesList) {
            switch (role) {
                case VIEWER -> roles.add(Role.ROLE_VIEWER);
                case PROJECT_ADMIN -> roles.add(Role.ROLE_PROJECT_ADMIN);
                case JNS_SENDER -> roles.add(Role.ROLE_JNS_SENDER);
                case EDITOR -> roles.add(Role.ROLE_EDITOR);
                case PUSHER -> roles.add(Role.ROLE_PUSHER);
                case SERVICE_PROVIDER_ADMIN -> roles.add(Role.ROLE_SERVICE_PROVIDER_ADMIN);
                case SERVICE_PROVIDER_ALERT_EDITOR -> roles.add(Role.ROLE_SERVICE_PROVIDER_ALERT_EDITOR);
                case BETA_TESTER -> roles.add(Role.ROLE_BETA_TESTER);
                case MUTER -> roles.add(Role.ROLE_MUTER);
                case ADMIN -> roles.add(Role.ROLE_MONITORING_ADMIN);
                case PROJECT_DELETER -> roles.add(Role.ROLE_MONITORING_PROJECT_DELETER);
                default -> throw new UnsupportedOperationException("Unsupported role: " + role);
            }
        }
        return roles;
    }

    public static AuthorizationObjects fromProtoResourcePath(List<ResourcePath> resourcePath) {
        return AuthorizationObjects.of(resourcePath.stream()
                .map(AccessDtoConverter::fromProto)
                .collect(Collectors.toSet()));
    }

    public static AuthorizationObject fromProto(ResourcePath resourcePath) {
        String projectId = null;
        String folderId = "";
        String serviceProviderId = "";
        String abcSlug = "";
        String abcRoleSlug = "";
        for (Resource resource : resourcePath.getPathList()) {
            if (resource.getType().equals(PROJECT_TYPE) || resource.getType().equals(CLOUD_TYPE)) {
                projectId = resource.getId();
            } else if (resource.getType().equals(FOLDER_TYPE)) {
                folderId = resource.getId();
            } else if (resource.getType().equals(SERVICE_PROVIDER_TYPE)) {
                serviceProviderId = resource.getId();
            } else if (resource.getType().equals(ABC_TYPE)) {
                abcSlug = resource.getId();
            } else if (resource.getType().equals(ABC_ROLE_TYPE)) {
                abcRoleSlug = resource.getId();
            }
        }
        if (!abcSlug.isEmpty() && !abcRoleSlug.isEmpty()) {
            return AuthorizationObject.abc(abcSlug, abcRoleSlug);
        }
        if (!serviceProviderId.isEmpty()) {
            return AuthorizationObject.serviceProvider(serviceProviderId, projectId);
        }
        if (projectId == null) {
            projectId = "";
        }
        if (projectId.isEmpty() && folderId.isEmpty()) {
            throw new BadRequestException("wrong resource path format: " + resourcePath);
        }
        return AuthorizationObject.classic(projectId, folderId);
    }

    public static AvailableResourcesResponse toProto(AuthorizationObjects objects) {
        List<ResourcePath> resources = new ArrayList<>(objects.objectIds().size());
        for (AuthorizationObject object : objects.objectIds()) {
            resources.add(toProto(object));
        }
        return AvailableResourcesResponse.newBuilder()
                .addAllResourcePath(resources.stream()
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build();
    }

    public static ResourcePath toProto(AuthorizationObject object) {
        if (object instanceof AuthorizationObject.ClassicAuthorizationObject cObject) {
            return ResourcePath.newBuilder()
                    .addPath(Resource.newBuilder()
                            .setType(cObject.projectId().isEmpty() ? FOLDER_TYPE : PROJECT_TYPE)
                            .setId(cObject.projectId().isEmpty() ? cObject.folderId() : cObject.projectId())
                            .build())
                    .build();
        } else if (object instanceof AuthorizationObject.SpAuthorizationObject spObject) {
            return ResourcePath.newBuilder()
                    .addPath(Resource.newBuilder()
                            .setType(SERVICE_PROVIDER_TYPE)
                            .setId(spObject.serviceProviderId())
                            .build())
                    .build();
        } else if (object instanceof AuthorizationObject.AbcAuthorizationObject authorizationObject) {
            return ResourcePath.newBuilder()
                    .addPath(Resource.newBuilder()
                            .setType(ABC_TYPE)
                            .setId(authorizationObject.abcSlug())
                            .build())
                    .addPath(Resource.newBuilder()
                            .setType(ABC_ROLE_TYPE)
                            .setId(authorizationObject.roleScopeSlug())
                            .build())
                    .build();
        }
        return null;
    }

    public static Subject toProto(AuthSubject subject) {
        switch (subject.getAuthType()) {
            case Anonymous -> {
                return Subject.newBuilder()
                        .setAnonymousAccount(Subject.AnonymousSubject.newBuilder().build())
                        .build();
            }
            case OAuth -> {
                return Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setUid(((OAuthSubject)subject).getUid())
                                .setLogin(((OAuthSubject)subject).getLogin())
                                .build())
                        .build();
            }
            case AsUser -> {
                return Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setLogin(((AsUserSubject)subject).getLogin())
                                .build())
                        .build();
            }
            case IAM -> {
                IamSubject iam = (IamSubject) subject;
                if (iam.isServiceAccount()) {
                    return Subject.newBuilder()
                            .setIamServiceSubject(Subject.IamServiceSubject.newBuilder()
                                    .setId(Objects.requireNonNull(iam.getId()))
                                    .build())
                            .build();
                } else if (iam.isUserAccount()) {
                    return Subject.newBuilder()
                            .setIamUserSubject(Subject.IamUserSubject.newBuilder()
                                    .setId(Objects.requireNonNull(iam.getId()))
                                    .build())
                            .build();
                }
            }
            case TvmUser -> {
                return Subject.newBuilder()
                        .setTvmUserSubject(Subject.TvmUserSubject.newBuilder()
                                .setUid(((TvmSubject.UserSubject)subject).getUid())
                                .setLogin(subject.getUniqueId())
                                .build())
                        .build();
            }
            case TvmService -> {
                return Subject.newBuilder()
                        .setTvmServiceSubject(Subject.TvmServiceSubject.newBuilder()
                                .setClientId(((TvmSubject.ServiceSubject)subject).getClientId())
                                .build())
                        .build();
            }
            case SessionIdCookie -> {
                return Subject.newBuilder()
                        .setSessionIdCookieSubject(Subject.SessionIdCookieSubject.newBuilder()
                                .setLogin(subject.getUniqueId())
                                .build())
                        .build();
            }
            case OpenId -> {
                return Subject.newBuilder()
                        .setOpenIdSubject(Subject.OpenIdSubject.newBuilder()
                                .setAccountId(((OpenIdSubject)subject).getAccountId())
                                .setLogin(((OpenIdSubject)subject).getLogin().orElse(""))
                                .build())
                        .build();
            }
        }
        throw new UnsupportedOperationException(subject + " not supported");
    }

    public static AuthorizeResponse toProto(Account account) {
        return AuthorizeResponse.newBuilder()
                .setAuthorizationType(account.getAuthorizationType().name())
                .addAllRoles(toProto(account.getRoles().toEnumSet()))
                .build();
    }

    public static AuthorizationType fromProto(String authorizationType) {
        try {
            return AuthorizationType.valueOf(authorizationType);
        } catch (Exception e) {
            return AuthorizationType.UNKNOWN;
        }
    }
}
