package ru.yandex.solomon.auth;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.monitoring.api.v3.project.manager.AvailableResourcesResponse;
import ru.yandex.monitoring.api.v3.project.manager.Resource;
import ru.yandex.monitoring.api.v3.project.manager.ResourcePath;
import ru.yandex.monitoring.api.v3.project.manager.Subject;
import ru.yandex.solomon.auth.dto.AccessDtoConverter;
import ru.yandex.solomon.auth.iam.IamSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.auth.tvm.TvmSubject;
import ru.yandex.solomon.core.exceptions.BadRequestException;

import static org.junit.Assert.assertEquals;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AccessDtoConverterTest {

    @Test
    public void fromProto_AnonymousSubject() {
        final Subject expected = Subject.newBuilder()
                .setAnonymousAccount(Subject.AnonymousSubject.newBuilder().build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals(AnonymousAuthSubject.INSTANCE, subject);
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_IamUserSubject() {
        final Subject expected = Subject.newBuilder()
                .setIamUserSubject(Subject.IamUserSubject.newBuilder()
                        .setId("1")
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", ((IamSubject) subject).getId());
    }

    @Test
    public void fromProto_IamServiceSubject() {
        final Subject expected = Subject.newBuilder()
                .setIamServiceSubject(Subject.IamServiceSubject.newBuilder()
                        .setId("1")
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", ((IamSubject) subject).getId());
    }

    @Test
    public void fromProto_TvmUserSubject() {
        final Subject expected = Subject.newBuilder()
                .setTvmUserSubject(Subject.TvmUserSubject.newBuilder()
                        .setLogin("1")
                        .setUid(11)
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", subject.getUniqueId());
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_TvmServiceSubject() {
        final Subject expected = Subject.newBuilder()
                .setTvmServiceSubject(Subject.TvmServiceSubject.newBuilder()
                        .setClientId(11)
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals(11, ((TvmSubject.ServiceSubject) subject).getClientId());
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_OAuthSubject() {
        final Subject expected = Subject.newBuilder()
                .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                        .setLogin("1")
                        .setUid(11)
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", ((OAuthSubject) subject).getLogin());
        assertEquals(11, ((OAuthSubject) subject).getUid());
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_OpenIdSubject() {
        final Subject expected = Subject.newBuilder()
                .setOpenIdSubject(Subject.OpenIdSubject.newBuilder()
                        .setLogin("1")
                        .setAccountId("2")
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", ((OpenIdSubject) subject).getLogin().get());
        assertEquals("2", ((OpenIdSubject) subject).getAccountId());
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_SessionIdCookieSubject() {
        final Subject expected = Subject.newBuilder()
                .setSessionIdCookieSubject(Subject.SessionIdCookieSubject.newBuilder()
                        .setLogin("1")
                        .build())
                .build();
        var subject = AccessDtoConverter.fromProto(expected);
        assertEquals("1", subject.getUniqueId());
        assertEquals(expected, AccessDtoConverter.toProto(subject));
    }

    @Test
    public void fromProto_resourcePathList() {
        var result = AccessDtoConverter.fromProtoResourcePath(List.of(
                ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.PROJECT_TYPE)
                                .setId("1")
                                .build())
                        .build(),
                ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.CLOUD_TYPE)
                                .setId("2")
                                .build())
                        .build()
        ));

        assertEquals(AuthorizationObjects.of(Set.of(
                AuthorizationObject.classic("1", ""),
                AuthorizationObject.classic("2", "")
        )),result);

    }

    @Test
    public void fromProto_ResourcePath() {
        var object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.PROJECT_TYPE)
                        .setId("1")
                        .build())
                .build());
        assertEquals(AuthorizationObject.classic("1", ""), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.CLOUD_TYPE)
                        .setId("1")
                        .build())
                .build());
        assertEquals(AuthorizationObject.classic("1", ""), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.CLOUD_TYPE)
                        .setId("1")
                        .build())
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.FOLDER_TYPE)
                        .setId("2")
                        .build())
                .build());
        assertEquals(AuthorizationObject.classic("1", "2"), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.FOLDER_TYPE)
                        .setId("1")
                        .build())
                .build());
        assertEquals(AuthorizationObject.classic("", "1"), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.SERVICE_PROVIDER_TYPE)
                        .setId("1")
                        .build())
                .build());
        assertEquals(AuthorizationObject.serviceProvider("1", null), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.SERVICE_PROVIDER_TYPE)
                        .setId("1")
                        .build())
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.PROJECT_TYPE)
                        .setId("2")
                        .build())
                .build());
        assertEquals(AuthorizationObject.serviceProvider("1", "2"), object);

        object = AccessDtoConverter.fromProto(ResourcePath.newBuilder()
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.ABC_TYPE)
                        .setId("1")
                        .build())
                .addPath(Resource.newBuilder()
                        .setType(AccessDtoConverter.ABC_ROLE_TYPE)
                        .setId("2")
                        .build())
                .build());
        assertEquals(AuthorizationObject.abc("1", "2"), object);

        try {
            AccessDtoConverter.fromProto(ResourcePath.newBuilder().build());
            throw new RuntimeException();
        } catch (BadRequestException e) {
        }
    }

    @Test
    public void toProto_ResourcePath() {
        LinkedHashSet<AuthorizationObject> set = new LinkedHashSet<>();
        set.add(AuthorizationObject.serviceProvider("1", "2"));
        set.add(AuthorizationObject.serviceProvider("1", null));
        set.add(AuthorizationObject.classic("", "1"));
        set.add(AuthorizationObject.classic("1", "2"));
        set.add(AuthorizationObject.classic("1", ""));
        set.add(AuthorizationObject.abc("1", "2"));
        var object = AccessDtoConverter.toProto(AuthorizationObjects.of(set));
        assertEquals(AvailableResourcesResponse.newBuilder()
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.SERVICE_PROVIDER_TYPE)
                                .setId("1")
                                .build())
                        .build())
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.SERVICE_PROVIDER_TYPE)
                                .setId("1")
                                .build())
                        .build())
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.FOLDER_TYPE)
                                .setId("1")
                                .build())
                        .build())
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.PROJECT_TYPE)
                                .setId("1")
                                .build())
                        .build())
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.PROJECT_TYPE)
                                .setId("1")
                                .build())
                        .build())
                .addResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.ABC_TYPE)
                                .setId("1")
                                .build())
                        .addPath(Resource.newBuilder()
                                .setType(AccessDtoConverter.ABC_ROLE_TYPE)
                                .setId("2")
                                .build())
                        .build())
                .build(), object);
    }

    @Test
    public void accountToProto() {
        var account = new Account("", AuthType.AsUser, AuthorizationType.ROLE, RoleSet.of(List.of()));
        var account2 = new Account("", AuthType.AsUser, AuthorizationType.UNKNOWN, RoleSet.of(List.of(Role.ADMIN)));
        var result = AccessDtoConverter.toProto(account);
        var result2 = AccessDtoConverter.toProto(account2);

        assertEquals("ROLE", result.getAuthorizationType());
        assertEquals("UNKNOWN", result2.getAuthorizationType());
        assertEquals(List.of(), result.getRolesList());
        assertEquals(List.of(ru.yandex.monitoring.api.v3.project.manager.Role.ROLE_MONITORING_ADMIN), result2.getRolesList());
    }

}
