package ru.yandex.solomon.auth;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class Account {

    public static final Account ANONYMOUS = new Account("", AuthType.Anonymous, AuthorizationType.UNKNOWN, RoleSet.EMPTY);

    private final String id;
    private final AuthType authType;
    private final AuthorizationType authorizationType;
    private final RoleSet roles;

    public Account(String id, AuthType authType, AuthorizationType authorizationType, RoleSet roles) {
        this.id = Objects.requireNonNull(id, "id");
        this.authType = Objects.requireNonNull(authType, "authType");
        this.authorizationType = Objects.requireNonNull(authorizationType, "authSource");
        this.roles = Objects.requireNonNull(roles, "roles");
    }

    public String getId() {
        return id;
    }

    public AuthType getAuthType() {
        return authType;
    }

    public AuthorizationType getAuthorizationType() {
        return authorizationType;
    }

    public RoleSet getRoles() {
        return roles;
    }

    public boolean can(Permission p) {
        return roles.hasPermission(p);
    }

    public boolean isAnonymous() {
        return this == ANONYMOUS || ("".equals(id) && authType == AuthType.Anonymous);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        Account account = (Account) o;
        if (authorizationType != account.authorizationType) {
            return false;
        }
        return id.equals(account.id);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    @Override
    public String toString() {
        return "Account{id='" + id + "', authType=" + authType + ", authSource=" + authorizationType + ", roles=" + roles + '}';
    }
}
