package ru.yandex.solomon.auth;

import java.util.Optional;

import io.grpc.Metadata;
import org.springframework.http.HttpHeaders;


/**
 * @author Sergey Polovko
 */
public enum AuthType {
    /**
     * For anonymous accounts only.
     */
    Anonymous("", ""),

    /**
     * Simple authentication mechanism in trusted (local) environment. Useful for debugging API
     * capabilities to check access rights.
     */
    AsUser(HttpHeaders.AUTHORIZATION, "AsUser "),

    /**
     * Use <a href="https://doc.yandex-team.ru/english/oauth/dg-internal">BlackBox</a> to validate tokens.
     */
    OAuth(HttpHeaders.AUTHORIZATION, "OAuth "),

    /**
     * Use <a href="https://cloud.yandex.ru/docs/iam/">Yandex.Cloud IAM</a> to validate tokens.
     */
    IAM(HttpHeaders.AUTHORIZATION, "Bearer "),

    /**
     * Use <a href="https://wiki.yandex-team.ru/passport/tvm2/">Ticket Vending Machine</a> to validate user tickets.
     */
    TvmUser("X-Ya-User-Ticket", ""),

    /**
     * Use <a href="https://wiki.yandex-team.ru/passport/tvm2/">Ticket Vending Machine</a> to validate service tickets.
     */
    TvmService("X-Ya-Service-Ticket", ""),

    /**
     * General authentication mechanism between web application and Solomon backend.
     * Use <a href="https://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml">sessionid method</a> to validate authentication cookies.
     */
    SessionIdCookie("", ""),

    /**
     * See more info at <a href="https://wiki.yandex-team.ru/cloud/iamrm/services/API-autentifikacii-Oblaka/">Cloud Authentication API</a>.
     */
    OpenId("", ""),
    ;

    private final String headerName;
    private final String valuePrefix;
    private final Optional<Metadata.Key<String>> metadataKey;

    AuthType(String headerName, String valuePrefix) {
        this.headerName = headerName;
        this.valuePrefix = valuePrefix;
        this.metadataKey = headerName.isEmpty()
            ? Optional.empty()
            : Optional.of(Metadata.Key.of(headerName, Metadata.ASCII_STRING_MARSHALLER));
    }

    public String getHeaderName() {
        return headerName;
    }

    public String getValuePrefix() {
        return valuePrefix;
    }

    public Metadata.Key<String> getMetadataKey() {
        return metadataKey.orElseThrow(() -> new RuntimeException("cannot get metadata key"));
    }
}
