package ru.yandex.solomon.auth;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.Attributes;
import io.grpc.Metadata;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.fake.AnonymousAuthenticator;


/**
 * @author Sergey Polovko
 */
public interface Authenticator {
    /**
     * Wraps given authenticator with transparent cache.
     */
    static Authenticator cache(Authenticator a) {
        return a instanceof AuthenticatorCache ? a : new AuthenticatorCache(a);
    }

    /**
     * Creates multiplexer to combine several authenticators.
     */
    static Authenticator mux(List<Pair<AuthType, Authenticator>> a) {
        return new AuthenticatorMux(a);
    }

    static Authenticator anonymous() {
        return new AnonymousAuthenticator();
    }

    /**
     * Retrieve authentication token from HTTP request.
     */
    Optional<AuthToken> getToken(ServerHttpRequest request);

    /**
     * Retrieve authentication token from gRPC request metadata headers.
     */
    Optional<AuthToken> getToken(Metadata headers, Attributes attributes);

    /**
     * Authenticates by token previously retrieved from HTTP request.
     *
     * In good case returns valid authentication subject (user info or service info). In case
     * of invalid authentication returned future completes with {@link AuthenticationException}.
     */
    CompletableFuture<AuthSubject> authenticate(AuthToken token);
}
