package ru.yandex.solomon.auth;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.ImmutableMap;
import io.grpc.Attributes;
import io.grpc.Metadata;
import io.opentracing.tag.Tags;
import io.opentracing.util.GlobalTracer;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;


/**
 * Authenticator multiplexer. Finds appropriate delegate authenticator implementation.
 *
 * @author Sergey Polovko
 */
final class AuthenticatorMux implements Authenticator {

    private final Authenticator[] authenticators;
    private final ImmutableMap<AuthType, Authenticator> authenticatorsMap;
    private final EnumMap<AuthType, AsyncMetrics> metrics;
    private final AsyncMetrics totalMetrics;

    AuthenticatorMux(List<Pair<AuthType, Authenticator>> authenticators) {
        this.authenticatorsMap = ImmutableMap.copyOf(authenticators);
        this.authenticators = authenticators.stream()
                .map(Map.Entry::getValue)
                .toArray(Authenticator[]::new);

        this.metrics = new EnumMap<>(AuthType.class);

        // register for all types to avoid nullability checking
        MetricRegistry registry = MetricRegistry.root();
        for (AuthType authType : AuthType.values()) {
            this.metrics.put(authType, new AsyncMetrics(registry, "auth", Labels.of("authType", authType.name())));
        }
        this.totalMetrics = new AsyncMetrics(registry, "auth", Labels.of("authType", "total"));
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        for (Authenticator a : authenticators) {
            Optional<AuthToken> token = a.getToken(request);
            if (token.isPresent()) {
                return token;
            }
        }
        return Optional.empty();
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        for (Authenticator a : authenticators) {
            Optional<AuthToken> token = a.getToken(headers, attributes);
            if (token.isPresent()) {
                return token;
            }
        }
        return Optional.empty();
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        long startMillis = System.currentTimeMillis();
        AsyncMetrics metrics = this.metrics.get(token.getType());
        metrics.callStarted();
        totalMetrics.callStarted();

        Authenticator a = authenticatorsMap.get(token.getType());
        if (a == null) {
            metrics.callCompletedError(0);
            totalMetrics.callCompletedError(0);
            String msg = token.getType() + " tokens are not supported";
            return CompletableFuture.failedFuture(new AuthenticationException(msg));
        }

        var tracer = GlobalTracer.get();
        var span = tracer.buildSpan("authenticate")
                .withTag("authType", token.getType().name())
                .start();

        try (var scope = tracer.activateSpan(span)) {
            return a.authenticate(token)
                    .whenComplete((s, t) -> {
                        long elapsedMillis = System.currentTimeMillis() - startMillis;
                        if (t != null) {
                            metrics.callCompletedError(elapsedMillis);
                            totalMetrics.callCompletedError(elapsedMillis);
                            span.setTag(Tags.ERROR, Boolean.TRUE);
                        } else {
                            metrics.callCompletedOk(elapsedMillis);
                            totalMetrics.callCompletedOk(elapsedMillis);
                            span.setTag(Tags.ERROR, Boolean.FALSE);
                        }
                        span.finish();
                    });
        }
    }
}
