package ru.yandex.solomon.auth;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.fake.AnonymousAuthorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public interface Authorizer {

    /**
     * Wraps given authorizer with transparent cache.
     */
    static Authorizer cache(Authorizer a) {
        return a instanceof AuthorizerCache ? a : new AuthorizerCache(a);
    }

    /**
     * Creates multiplexer to combine several authorizers.
     */
    static Authorizer mux(List<Authorizer> a) {
        return new AuthorizerMux(a);
    }

    /**
     * Creates "one-of" complex authorizer
     */
    static Authorizer oneOf(Authorizer first, Authorizer second) {
        return new AuthorizerOneOf(first, second);
    }

    /**
     * Creates anonymous authorizer, which will authorize all requests.
     */
    static Authorizer anonymous() {
        return new AnonymousAuthorizer();
    }

    /**
     * Checks ability to use this authorizer to authorize subject.
     */
    boolean canAuthorize(AuthSubject subject);

    /**
     * Checks ability to use this authorizer to authorize this object.
     */
    boolean canAuthorizeObject(AuthorizationObject object);

    /**
     * @return successfully completed future if permission is granted for given subject in given project
     *         otherwise exceptionally {@link AuthorizationException} completed future.
     */
    default CompletableFuture<Account> authorize(
        AuthSubject subject,
        String projectId,
        String folderId,
        Permission permission)
    {
        return authorize(subject, AuthorizationObject.classic(projectId, folderId), permission);
    }

    /**
     * @return successfully completed future if permission is granted for given subject
     *         otherwise exceptionally {@link AuthorizationException} completed future.
     */
    default CompletableFuture<Account> authorize(AuthSubject subject, String projectId, Permission permission) {
        return authorize(subject, projectId, "", permission);
    }

    /**
     * @return successfully completed future if permission is granted for given subject in given object
     *         otherwise exceptionally {@link AuthorizationException} completed future.
     */
    CompletableFuture<Account> authorize(
            AuthSubject subject,
            AuthorizationObject authorizationObject,
            Permission permission);

    default CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        return CompletableFuture.completedFuture(AuthorizationObjects.EMPTY);
    }
}
