package ru.yandex.solomon.auth;

import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;


/**
 * @author Sergey Polovko
 */
final class AuthorizerCache implements Authorizer {
    private static final Logger logger = LoggerFactory.getLogger(AuthorizerCache.class);

    private static final long CACHE_POSITIVE_TTL_MINUTES = 10;
    private static final long CACHE_NEGATIVE_TTL_MINUTES = 1;
    private static final long CACHE_MAX_SIZE = 6000;

    private final Authorizer delegate;
    private final InnerCache<AuthorizationObjectsKey, AuthorizationObjects> authorizationObjectsCache;
    private final InnerCache<Key, Account> authorizationCache;

    AuthorizerCache(Authorizer delegate) {
        this.delegate = delegate;
        this.authorizationObjectsCache = new InnerCache<>("authorizer.authorization_objects.", CACHE_NEGATIVE_TTL_MINUTES, CACHE_NEGATIVE_TTL_MINUTES);
        this.authorizationCache = new InnerCache<>("authorizer.");
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return delegate.canAuthorize(subject);
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return delegate.canAuthorizeObject(object);
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        AuthorizationObjectsKey key = new AuthorizationObjectsKey(subject, roles);
        return authorizationObjectsCache.execute(key, () -> delegate.getAvailableAuthorizationObjects(subject, roles, types));
    }

    @Override
    public CompletableFuture<Account> authorize(
            AuthSubject subject,
            AuthorizationObject authorizationObject,
            Permission permission)
    {
        Key key = new Key(subject, authorizationObject, permission);
        return authorizationCache.execute(key, () -> delegate.authorize(subject, authorizationObject, permission));
    }

    private static final class InnerCache<Key, Value> {
        private final Cache<Key, Value> positive;
        private final Cache<Key, Throwable> negative;
        private final Rate cacheHitOk;
        private final Rate cacheHitError;
        private final Rate cacheMiss;

        InnerCache(String metricsPrefix) {
            this(metricsPrefix, CACHE_POSITIVE_TTL_MINUTES, CACHE_NEGATIVE_TTL_MINUTES);
        }

        public InnerCache(String metricsPrefix, long cachePositiveTtlMinutes, long cacheNegativeTtlMinutes) {
            this.positive = CacheBuilder.newBuilder()
                    .expireAfterWrite(cachePositiveTtlMinutes, TimeUnit.MINUTES)
                    .maximumSize(CACHE_MAX_SIZE)
                    .build();
            this.negative = CacheBuilder.newBuilder()
                    .expireAfterWrite(cacheNegativeTtlMinutes, TimeUnit.MINUTES)
                    .maximumSize(CACHE_MAX_SIZE)
                    .build();
            this.cacheHitOk = MetricRegistry.root().rate(metricsPrefix + "cacheHitOk");
            this.cacheHitError = MetricRegistry.root().rate(metricsPrefix + "cacheHitError");
            this.cacheMiss = MetricRegistry.root().rate(metricsPrefix + "cacheMiss");
        }

        CompletableFuture<Value> execute(Key key, Supplier<CompletableFuture<Value>> supplier) {
            var value = positive.getIfPresent(key);
            if (value != null) {
                cacheHitOk.inc();
                return CompletableFuture.completedFuture(value);
            }

            Throwable error = negative.getIfPresent(key);
            if (error != null) {
                cacheHitError.inc();
                String msg = "previous authorization error is not yet expired" + (StringUtils.isEmpty(error.getMessage()) ? "" : ": " + error.getMessage());
                return CompletableFuture.failedFuture(new AuthorizationException(msg, error));
            }

            cacheMiss.inc();
            return CompletableFutures.safeCall(supplier::get)
                    .whenComplete((a, t) -> {
                        if (t != null) {
                            Throwable cause = CompletableFutures.unwrapCompletionException(t);
                            logger.warn("cannot authorize {}", key, cause);
                            negative.put(key, cause);
                        } else {
                            positive.put(key, a);
                        }
                    });
        }
    }

    /**
     * CACHE KEY
     */
    private static final record Key(AuthSubject subject, AuthorizationObject authorizationObject,
                                    Permission permission) {
    }

    /**
     * CACHE KEY for AuthorizationObjects
     */
    private static final record AuthorizationObjectsKey(AuthSubject subject, Set<Role> roles) {
    }
}
