package ru.yandex.solomon.auth;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import com.google.common.collect.ImmutableList;
import io.opentracing.tag.Tags;
import io.opentracing.util.GlobalTracer;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * Authorizer multiplexer. Finds appropriate delegate authorizer implementation.
 *
 * @author Sergey Polovko
 */
final class AuthorizerMux implements Authorizer {

    private final ImmutableList<Authorizer> authorizers;
    private final MetricRegistry registry;
    private final EnumMap<AuthType, AsyncMetrics> metrics;
    private final AsyncMetrics totalMetrics;
    private final ConcurrentHashMap<Labels, Rate> requestsByObjectAndType;
    private final HashMap<RequestTypeKey, Rate> requestsByType;

    AuthorizerMux(List<Authorizer> authorizers) {
        this.authorizers = ImmutableList.copyOf(authorizers);
        MetricRegistry registry = MetricRegistry.root();
        this.registry = registry;
        this.metrics = new EnumMap<>(AuthType.class);
        for (AuthType authType : AuthType.values()) {
            this.metrics.put(authType, new AsyncMetrics(registry, "authorizer", Labels.of("authType", authType.name())));
        }
        this.totalMetrics = new AsyncMetrics(registry, "authorizer", Labels.of("authType", "total"));
        this.requestsByType = new HashMap<>();
        for (AuthorizationType type : AuthorizationType.values()) {
            for (var authObjectType : AuthorizationObject.Type.values()) {
                Labels labels = Labels.of(authObjectType.getIdName(), "total", "type", type.name());
                this.requestsByType.put(new RequestTypeKey(type, authObjectType.getIdName()), registry.rate("authorizer." + authObjectType.getName() + ".requests", labels));
            }
        }
        this.requestsByObjectAndType = new ConcurrentHashMap<>(1000);
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        for (Authorizer a : authorizers) {
            if (a.canAuthorize(subject)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        for (Authorizer a : authorizers) {
            if (a.canAuthorizeObject(object)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        List<CompletableFuture<AuthorizationObjects>> futures = new ArrayList<>();
        for (Authorizer authorizer : authorizers) {
            futures.add(authorizer.getAvailableAuthorizationObjects(subject, roles, types));
        }
        return CompletableFutures.allOf(futures)
                .thenApply(authorizationObjects -> {
                    var result = AuthorizationObjects.EMPTY;
                    for (var authorizationObject : authorizationObjects) {
                        result = result.combine(authorizationObject);
                    }
                    return result;
                });
    }

    @Override
    public CompletableFuture<Account> authorize(
            AuthSubject subject,
            AuthorizationObject authorizationObject,
            Permission permission)
    {
        var tracer = GlobalTracer.get();
        var span = tracer.buildSpan("authorize")
                .withTag("subject", subject.toString())
                .start();

        long startMillis = System.currentTimeMillis();
        AsyncMetrics metrics = this.metrics.get(subject.getAuthType());
        metrics.callStarted();
        totalMetrics.callStarted();

        try (var scope = tracer.activateSpan(span)) {
            for (Authorizer a : authorizers) {
                if (a.canAuthorize(subject) && a.canAuthorizeObject(authorizationObject)) {
                    return a.authorize(subject, authorizationObject, permission)
                            .whenComplete((r, e) -> {
                                long elapsedMillis = System.currentTimeMillis() - startMillis;
                                if (e != null) {
                                    metrics.callCompletedError(elapsedMillis);
                                    totalMetrics.callCompletedOk(elapsedMillis);
                                } else {
                                    metrics.callCompletedOk(elapsedMillis);
                                    totalMetrics.callCompletedOk(elapsedMillis);
                                    AuthorizationType authorizationType = r.getAuthorizationType();
                                    registerAuthorizationType(authorizationObject, authorizationType);
                                }
                                span.setTag(Tags.ERROR, e != null).finish();
                            });
                }
            }

            totalMetrics.callCompletedError(0);
            span.setTag(Tags.ERROR, Boolean.TRUE).finish();
            return failedFuture(new AuthorizationException("cannot authorize: " + subject));
        }
    }

    private void registerAuthorizationType(AuthorizationObject authorizationObject, AuthorizationType authorizationType) {
        var objectValue = "";
        if (authorizationObject instanceof AuthorizationObject.ClassicAuthorizationObject co) {
            objectValue = co.projectId();
        } else  if (authorizationObject instanceof AuthorizationObject.SpAuthorizationObject so) {
            objectValue = so.serviceProviderId();
        }
        if (!objectValue.isEmpty() && StrictValidator.SELF.isValueValid(objectValue)) {
            requestsByType.get(new RequestTypeKey(authorizationType, authorizationObject.getType().getIdName())).inc();
            Labels labels = Labels.of(authorizationObject.getType().getIdName(), objectValue, "type", authorizationType.name());
            Rate rate = requestsByObjectAndType.computeIfAbsent(labels, ignored -> registry.rate("authorizer." + authorizationObject.getType().getName() + ".requests", labels));
            rate.inc();
        }
    }

    private record RequestTypeKey(AuthorizationType authorizationType, String objectName) {
    }
}
