package ru.yandex.solomon.auth;

import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;

/*
 * Special authorizer support intersecting authorizers
 *
 * @author Oleg Baryshnikov
 */
final class AuthorizerOneOf implements Authorizer {
    private final Authorizer first;
    private final Authorizer second;

    AuthorizerOneOf(Authorizer first, Authorizer second) {
        this.first = first;
        this.second = second;
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return first.canAuthorize(subject) || second.canAuthorize(subject);
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return first.canAuthorizeObject(object) || second.canAuthorizeObject(object);
    }

    @Override
    public CompletableFuture<AuthorizationObjects> getAvailableAuthorizationObjects(AuthSubject subject, Set<Role> roles, EnumSet<AuthorizationObject.Type> types) {
        return first.getAvailableAuthorizationObjects(subject, roles, types)
                .thenCompose(authorizationObjects -> {
                    return second.getAvailableAuthorizationObjects(subject, roles, types)
                            .thenApply(featureObjects -> featureObjects.combine(authorizationObjects));
                });
    }

    @Override
    public CompletableFuture<Account> authorize(AuthSubject subject, AuthorizationObject authorizationObject, Permission permission) {
        if (first.canAuthorize(subject)) {
            return first.authorize(subject, authorizationObject, permission)
                .handle((Pair::of))
                .thenCompose(pair -> {
                   if (pair.getRight() != null) {
                       return second.authorize(subject, authorizationObject, permission);
                   }
                   return CompletableFuture.completedFuture(pair.getLeft());
                });
        }

        return second.authorize(subject, authorizationObject, permission);
    }
}
