package ru.yandex.solomon.auth;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;

/**
 * @author Oleg Baryshnikov
 * @author Sergey Polovko
 */
public class SolomonTeam {
    private static final Logger logger = LoggerFactory.getLogger(SolomonTeam.class);
    private static final Set<String> DISABLED_USERS = Collections.newSetFromMap(new ConcurrentHashMap<>());

    public static boolean isMember(AuthSubject subject) {
        return isMember(subject, false);
    }

    public static boolean isMember(AuthSubject subject, boolean includeDisabledUsers) {
        if (isMember(subject.getUniqueId(), includeDisabledUsers)) {
            return true;
        }
        Optional<String> login = AuthSubject.getLogin(subject);
        return login.isPresent() && isMember(login.get(), includeDisabledUsers);
    }

    public static boolean isMember(String login) {
        return isMember(login, false);
    }

    private static boolean isMember(String login, boolean includeDisabledUsers) {
        return (includeDisabledUsers || !DISABLED_USERS.contains(login)) && TeamHolder.TEAM.contains(login);
    }

    /**
     * Parses list of strings with skipping line comments (started with {@code #}) and empty lines.
     */
    static ImmutableSet<String> parse(String content) {
        var builder = ImmutableSet.<String>builder();

        for (String line : StringUtils.split(content, '\n')) {
            int commentIdx = line.indexOf('#');
            if (commentIdx != -1) {
                line = line.substring(0, commentIdx);
            }
            line = StringUtils.strip(line);
            if (!line.isEmpty()) {
                builder.add(line);
            }
        }

        return builder.build();
    }

    /**
     * Safely read and parse content of the file with team members list.
     */
    private static ImmutableSet<String> safeReadFrom(Path path) {
        final String content;
        try {
            content = Files.readString(path, Charsets.UTF_8);
        } catch (Throwable e) {
            // do not silently eat exception
            // (logger may not be initialized yet)
            e.printStackTrace();
            logger.error("Failed read from: {}", path, e);
            return ImmutableSet.of(System.getProperty("user.name"));
        }
        return parse(content);
    }

    private static final class TeamHolder {
        static final ImmutableSet<String> TEAM = safeReadFrom(Path.of("/etc/solomon/team"));
    }

    public static class DisabledUsersManager {
        @ManagerMethod
        public void disable(String login) {
            if (!TeamHolder.TEAM.contains(login)) {
                throw new IllegalArgumentException("login doesn't exist");
            }
            DISABLED_USERS.add(login);
        }

        @ManagerMethod
        public void enable(String login) {
            DISABLED_USERS.remove(login);
        }

        @ManagerMethod
        public Set<String> list() {
            return DISABLED_USERS;
        }
    }
}
