package ru.yandex.solomon.auth.grpc;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.ForwardingServerCall;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.GrpcUtils;
import ru.yandex.solomon.auth.AuthSubject;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class GrpcRequestLoggingInterceptor implements ServerInterceptor {
    private static final Logger logger = LoggerFactory.getLogger("access");

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(
            ServerCall<ReqT, RespT> call,
            Metadata headers,
            ServerCallHandler<ReqT, RespT> next)
    {
        String method = call.getMethodDescriptor().getFullMethodName();
        String userIp = GrpcUtils.realOrRemoteIp(call.getAttributes());
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubjectOrNull(Context.current());
        long startMillis = System.currentTimeMillis();
        var prefix = format(userIp, authSubject, method);

        logger.info(prefix + "...");

        return next.startCall(new ForwardingServerCall.SimpleForwardingServerCall<>(call) {
            @Override
            public void close(Status status, Metadata trailers) {
                super.close(status, trailers);
                long elapsedMillis = System.currentTimeMillis() - startMillis;
                logger.info(prefix + " " + status.getCode() + " " + elapsedMillis + " ms");
            }
        }, headers);
    }

    public String format(String userIp, @Nullable AuthSubject authSubject, String method) {
        String maybeAuthSubject = authSubject != null ? authSubject.toString() : "";
        return userIp + " " + maybeAuthSubject + " " + method;
    }

}

