package ru.yandex.solomon.auth.http;

import io.grpc.Metadata;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;


/**
 * @author Sergey Polovko
 */
public final class AuthHeaderHelper {

    private AuthHeaderHelper() {
    }

    public static boolean hasValidHeader(AuthType type, ServerHttpRequest request) {
        String header = request.getHeaders().getFirst(type.getHeaderName());
        return header != null && header.startsWith(type.getValuePrefix());
    }

    public static boolean hasValidHeader(AuthType type, Metadata headers) {
        String header = headers.get(type.getMetadataKey());
        return header != null && header.startsWith(type.getValuePrefix());
    }

    public static String getAuthToken(AuthType type, ServerHttpRequest request) {
        String header = request.getHeaders().getFirst(type.getHeaderName());
        if (header == null) {
            throw headerIsMissing(type);
        }
        return getTokenValue(type, header);
    }

    public static String getAuthToken(AuthType type, Metadata headers) {
        String header = headers.get(type.getMetadataKey());
        if (header == null) {
            throw new AuthenticationException(type.getHeaderName() + " header is missing");
        }
        return getTokenValue(type, header);
    }

    public static String getOAuthToken(String authHeader) {
        return getTokenValue(AuthType.OAuth, authHeader);
    }

    public static String getIamToken(String authHeader) {
        return getTokenValue(AuthType.IAM, authHeader);
    }

    private static String getTokenValue(AuthType type, String header) {
        String prefix = type.getValuePrefix();
        if (!header.startsWith(prefix)) {
            String message = String.format("%s header value must starts with '%s'", type.getHeaderName(), prefix);
            throw new AuthenticationException(message);
        }
        return header.substring(prefix.length());
    }

    private static AuthenticationException headerIsMissing(AuthType type) {
        return new AuthenticationException(type.getHeaderName() + " header is missing");
    }
}
