package ru.yandex.solomon.auth.http;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.springframework.core.MethodParameter;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.reactive.BindingContext;
import org.springframework.web.reactive.result.method.HandlerMethodArgumentResolver;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.selfmon.trace.SpanAwareFuture;

import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class AuthMethodArgumentResolver implements HandlerMethodArgumentResolver {

    public static final String AUTH_SUBJECT_ATTRIBUTE = AuthSubject.class.getCanonicalName();
    private final HttpAuthenticator authenticator;

    public AuthMethodArgumentResolver(HttpAuthenticator authenticator) {
        this.authenticator = authenticator;
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return parameter.getParameterAnnotation(RequireAuth.class) != null
                || parameter.getParameterAnnotation(OptionalAuth.class) != null;
    }

    @Override
    public Mono<Object> resolveArgument(
            MethodParameter parameter,
            BindingContext bindingContext,
            ServerWebExchange exchange)
    {
        boolean isOptional = parameter.getParameterAnnotation(OptionalAuth.class) != null;
        ServerHttpRequest request = exchange.getRequest();
        var future = SpanAwareFuture.wrap(safeCall(() -> authenticator.authenticate(request, isOptional)))
                .thenApply(authSubject -> {
                    exchange.getAttributes().put(AUTH_SUBJECT_ATTRIBUTE, authSubject.toString());
                    return authSubject;
                })
                .exceptionally(e -> handleException(e, isOptional));
        return Mono.fromFuture(future);
    }

    private AuthSubject handleException(Throwable e, boolean isOptional) {
        if (isOptional) {
            var cause = CompletableFutures.unwrapCompletionException(e);
            if (cause instanceof AuthenticationException) {
                // treat invalid auth as anonymous access in case authentication is optional
                return AnonymousAuthSubject.INSTANCE;
            }
        }

        Throwables.throwIfUnchecked(e);
        throw new RuntimeException(e);
    }
}
