package ru.yandex.solomon.auth.http;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class HttpAuthenticator {
    private final Authenticator authenticator;

    @Autowired
    public HttpAuthenticator(Authenticator authenticator) {
        this.authenticator = authenticator;
    }

    public CompletableFuture<AuthSubject> authenticate(ServerHttpRequest request) {
        return authenticate(request, false);
    }

    @SuppressWarnings("unused")
    public CompletableFuture<AuthSubject> authenticateOptionally(ServerHttpRequest request) {
        return authenticate(request, true);
    }

    CompletableFuture<AuthSubject> authenticate(ServerHttpRequest request, boolean isOptional) {
        Optional<AuthToken> token = authenticator.getToken(request);
        if (token.isEmpty()) {
            if (isOptional) {
                // We can return fake authentication subject in case of unrequired authentication
                // for old REST API or other unauthenticated endpoints
                return CompletableFuture.completedFuture(AnonymousAuthSubject.INSTANCE);
            }

            return CompletableFuture.failedFuture(new AuthenticationException("cannot authenticate request", true));
        }

        return authenticator.authenticate(token.get());
    }
}
