package ru.yandex.solomon.auth.iam;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.Attributes;
import io.grpc.Metadata;
import org.springframework.http.server.reactive.ServerHttpRequest;
import yandex.cloud.auth.api.AsyncCloudAuthClient;
import yandex.cloud.auth.api.credentials.IamToken;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.http.AuthHeaderHelper;


/**
 * @author Sergey Polovko
 */
public class IamAuthenticator implements Authenticator {

    private final AsyncCloudAuthClient authClient;

    public IamAuthenticator(AsyncCloudAuthClient authClient) {
        this.authClient = authClient;
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        if (AuthHeaderHelper.hasValidHeader(AuthType.IAM, request)) {
            String token = AuthHeaderHelper.getAuthToken(AuthType.IAM, request);
            return Optional.of(new AuthToken(AuthType.IAM, token));
        }
        return Optional.empty();
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        if (AuthHeaderHelper.hasValidHeader(AuthType.IAM, headers)) {
            String token = AuthHeaderHelper.getAuthToken(AuthType.IAM, headers);
            return Optional.of(new AuthToken(AuthType.IAM, token));
        }
        return Optional.empty();
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        try {
            return authClient.authenticate(new IamToken(token.getValue()))
                .handle((subject, throwable) -> {
                    if (throwable == null) {
                        return new IamSubject(subject);
                    }
                    var cause = CompletableFutures.unwrapCompletionException(throwable);
                    String message = "cannot authenticate by IAM token, cause: " + cause.getMessage();
                    throw new AuthenticationException(message, cause);
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }
}
