package ru.yandex.solomon.auth.iam;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;
import yandex.cloud.auth.api.AsyncCloudAuthClient;
import yandex.cloud.auth.api.Resource;
import yandex.cloud.auth.api.Subject;
import yandex.cloud.auth.api.Subject.UserAccount;
import yandex.cloud.auth.api.exception.CloudAuthPermissionDeniedException;
import yandex.cloud.auth.api.exception.CloudAuthUnauthenticatedException;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.util.async.InFlightLimiter;


/**
 * @author Sergey Polovko
 */
public class IamAuthorizer implements Authorizer {

    private final AsyncCloudAuthClient authClient;
    private final InFlightLimiter limiter;

    public IamAuthorizer(AsyncCloudAuthClient authClient, int maxInflight) {
        this.authClient = authClient;
        this.limiter = new InFlightLimiter(maxInflight != 0 ? maxInflight : 50);
    }

    @Override
    public boolean canAuthorize(AuthSubject subject) {
        return subject instanceof IamSubject || subject instanceof OpenIdSubject;
    }

    @Override
    public boolean canAuthorizeObject(AuthorizationObject object) {
        return object.getType() == AuthorizationObject.Type.CLASSIC || object.getType() == AuthorizationObject.Type.SERVICE_PROVIDER;
    }

    @Override
    public CompletableFuture<Account> authorize(
        AuthSubject subject,
        AuthorizationObject authorizationObject,
        Permission permission)
    {
        try {
            if (authorizationObject instanceof AuthorizationObject.SpAuthorizationObject object) {
                if (StringUtils.isEmpty(object.objectId())) {
                    // todo (alextrushkin) need to auth service provider as resource in iam SOLOMON-7985
                    return CompletableFuture.completedFuture(new Account(subject.getUniqueId(), AuthType.IAM, AuthorizationType.IAM, RoleSet.EMPTY));
                }
                Resource resource = Resource.cloud(object.objectId());
                return authorize(subject, permission, resource);
            } else if (authorizationObject instanceof AuthorizationObject.ClassicAuthorizationObject object) {
                Resource resource = object.folderId().isEmpty()
                        ? Resource.cloud(object.projectId())
                        : Resource.folder(object.folderId());

                return authorize(subject, permission, resource);
            }
            return CompletableFuture.failedFuture(new RuntimeException("unsupported auth object " + authorizationObject));
        } catch (Throwable x) {
            return CompletableFuture.failedFuture(x);
        }
    }

    private CompletableFuture<Account> authorize(AuthSubject subject, Permission permission, Resource resource) {
        final Subject.Id id;
        if (subject instanceof IamSubject) {
            id = ((IamSubject) subject).getSubject().toId();
        } else if (subject instanceof OpenIdSubject) {
            id = new UserAccount.Id(((OpenIdSubject) subject).getAccountId());
        } else {
            return CompletableFuture.failedFuture(new RuntimeException("unsupported auth subject " + subject));
        }

        var result = new CompletableFuture<Account>();
        limiter.run(() -> authClient.authorize(id, permission.getSlug(), resource)
                .whenComplete((s, t) -> {
                    try {
                        if (t != null) {
                            result.completeExceptionally(handleException(t));
                        } else {
                            result.complete(new Account(subject.getUniqueId(), AuthType.IAM, AuthorizationType.IAM, RoleSet.EMPTY));
                        }
                    } catch (Throwable x) {
                        result.completeExceptionally(x);
                    }
                }));
        return result;
    }

    private AuthorizationException handleException(@Nonnull Throwable throwable) {
        throwable = CompletableFutures.unwrapCompletionException(throwable);
        var cause = throwable;
        do {
            if (cause instanceof CloudAuthPermissionDeniedException) {
                return new AuthorizationException(cause.getMessage());
            } else if (cause instanceof CloudAuthUnauthenticatedException) {
                return new AuthorizationException(cause.getMessage());
            }

            cause = cause.getCause();
        } while (cause != null);
        return new AuthorizationException(throwable.getMessage(), throwable);
    }
}
