package ru.yandex.solomon.auth.iam;

import java.util.Objects;

import javax.annotation.Nullable;

import yandex.cloud.auth.api.Subject;
import yandex.cloud.auth.api.Subject.ServiceAccount;
import yandex.cloud.auth.api.Subject.UserAccount;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;


/**
 * @author Sergey Polovko
 */
public class IamSubject implements AuthSubject {

    private final Subject subject;

    public IamSubject(Subject subject) {
        this.subject = Objects.requireNonNull(subject, "subject");
        if (subject instanceof Subject.Anonymous) {
            throw new AuthenticationException("anonymous subject is not supported for IAM");
        }
    }

    @Override
    public String getUniqueId() {
        String id = getId();
        if (id == null) {
            throw new IllegalStateException("unsupported subject type: " + subject.getClass());
        }
        return id;
    }

    @Override
    public AuthType getAuthType() {
        return AuthType.IAM;
    }

    @Nullable
    public String getId() {
        if (isServiceAccount()) {
            return ((ServiceAccount.Id) subject.toId()).getId();
        }
        if (isUserAccount()) {
            return ((UserAccount.Id) subject.toId()).getId();
        }
        if (isSimpleAccount()) {
            return ((Subject.NamedId) subject.toId()).getId();
        }
        return null;
    }

    @Nullable
    public String getFolderId() {
        if (isServiceAccount()) {
            return ((ServiceAccount) subject).getFolderId();
        }
        return null;
    }

    public Subject getSubject() {
        return subject;
    }

    public boolean isUserAccount() {
        return subject instanceof UserAccount;
    }

    public boolean isServiceAccount() {
        return subject instanceof ServiceAccount;
    }

    private boolean isSimpleAccount() {
        return subject instanceof SimpleSubject;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        IamSubject that = (IamSubject) o;
        return Objects.equals(getId(), that.getId());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(128);
        sb.append("IamSubject{");
        if (isUserAccount()) {
            sb.append("type=user");
        } else if (isServiceAccount()) {
            sb.append("type=service");
        } else if (isSimpleAccount()) {
            sb.append("type=simple");
        } else {
            sb.append("type=unknown");
        }
        sb.append(", id=").append(getId());
        if (isServiceAccount()) {
            sb.append(", folderId=").append(getFolderId());
        }
        sb.append('}');
        return sb.toString();
    }

    public static record SimpleSubject(Subject.Id id) implements Subject {

        @Override
        public Id toId() {
            return id;
        }
    }
}
