package ru.yandex.solomon.auth.iam;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import com.google.common.net.HostAndPort;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.cloud.token.grpc.GrpcIamTokenClient;
import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.solomon.config.IamKeyJson;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;

public class IamTokenContext {

    @Bean(name = "authIamTokenClient")
    IamTokenClient iamTokenClient(TAuthConfig config, ThreadPoolProvider threads) {
        if (!config.hasIamConfig() || config.getIamConfig().getTokenServiceAddresses().equals("")) {
            return null;
        }
        ExecutorService executor = threads.getExecutorService(
                config.getIamConfig().getThreadPoolName(),
                "IamTokenClient.ThreadPoolName");
        HostAndPort address = DiscoveryServices.resolve(config.getIamConfig().getTokenServiceAddresses()).get(0);
        return new GrpcIamTokenClient(IamTokenClientOptions.forAddress(address.getHost(), address.getPort())
                .withHandlerExecutor(executor)
                .withUserAgent("Solomon"));
    }

    @Bean(name = "authContextIamTokenProvider")
    TokenProvider tokenProvider(
            @Qualifier("authIamTokenClient") Optional<IamTokenClient> iamTokenClient,
            TAuthConfig authConfig,
            ThreadPoolProvider threads,
            SecretProvider secretProvider)
    {
        if (!authConfig.hasIamConfig()) {
            return TokenProvider.of("fake-token");
        }

        if (!authConfig.getIamConfig().hasKeyAuth()) {
            return TokenProvider.of("fake-token");
        }

        if (iamTokenClient.isEmpty()) {
            return TokenProvider.of("fake-token");
        }

        var jwtBuilder = Jwt.newBuilder()
                .withTtl(Duration.ofHours(1));

        switch (authConfig.getIamConfig().getKeyTypeCase()) {
            case KEY_AUTH -> {
                var keyAuth = authConfig.getIamConfig().getKeyAuth();
                var keyFilePath = Path.of(keyAuth.getKeyPath());
                if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
                    return TokenProvider.of("fake-token");
                }
                jwtBuilder.withAccountId(keyAuth.getAccountId())
                        .withKeyId(keyAuth.getKeyId())
                        .withPrivateKey(keyFilePath);
            }
            case KEY_JSON_AUTH -> {
                Optional<String> keyData = secretProvider.getSecret(authConfig.getIamConfig().getKeyJsonAuth().getKeyData());
                if (SolomonEnv.DEVELOPMENT.isActive() && keyData.isEmpty()) {
                    return TokenProvider.of("fake-token");
                }

                IamKeyJson iamKey = IamKeyJson.parseFromJson(
                        keyData.orElseThrow(() -> new RuntimeException("cannot create auth provider with empty private key")));

                jwtBuilder.withAccountId(iamKey.getAccountId())
                        .withKeyId(iamKey.getId())
                        .withPrivateKey(iamKey.getPrivateKey());
            }
        }
        return TokenProvider.iam(iamTokenClient.get(), jwtBuilder, threads.getSchedulerExecutorService());
    }
}
