package ru.yandex.solomon.auth.internal;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.RoleSet;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Sergey Polovko
 */
public class InternalAuthorizer {

    private final Set<String> allowed;

    public InternalAuthorizer(Set<String> allowed) {
        this.allowed = allowed;
    }

    public CompletableFuture<Account> authorize(AuthSubject subject) {
        // Directly InternalAuthorizer checks only Solomon Team users including disabled users.
        // It's necessary to test authorization as out-staff user and return back using StaffOnly UI.
        return authorize(subject, true);
    }

    private CompletableFuture<Account> authorize(AuthSubject subject, boolean includeDisabledUsers) {
        if (allowed.contains(subject.getUniqueId()) || SolomonTeam.isMember(subject, includeDisabledUsers)) {
            return completedFuture(new Account(subject.getUniqueId(), subject.getAuthType(), AuthorizationType.INTERNAL, RoleSet.ALL));
        }

        return failedFuture(new AuthorizationException("cannot authorize subject: " + subject));
    }

    public Authorizer adapter() {
        return new Authorizer() {
            @Override
            public boolean canAuthorize(AuthSubject subject) {
                return true;
            }

            @Override
            public boolean canAuthorizeObject(AuthorizationObject object) {
                return true;
            }

            @Override
            public CompletableFuture<Account> authorize(
                    AuthSubject subject,
                    AuthorizationObject authorizationObject,
                    Permission permission)
            {
                // InternalAuthorizer as part of common authorizer (see AuthContext), so we ignore disabled users.
                return InternalAuthorizer.this.authorize(subject, false);
            }
        };
    }
}
