package ru.yandex.solomon.auth.oauth;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.Attributes;
import io.grpc.Metadata;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.grpc.utils.GrpcUtils;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.http.AuthHeaderHelper;
import ru.yandex.solomon.util.http.HttpUtils;


/**
 * @author Sergey Polovko
 */
public class OAuthAuthenticator implements Authenticator {
    private final BlackboxClient blackboxClient;

    public OAuthAuthenticator(BlackboxClient blackboxClient) {
        this.blackboxClient = blackboxClient;
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        if (AuthHeaderHelper.hasValidHeader(AuthType.OAuth, request)) {
            String token = AuthHeaderHelper.getAuthToken(AuthType.OAuth, request);
            String userIp = HttpUtils.realOrRemoteIp(request);
            return getOAuthToken(token, userIp);
        }
        return Optional.empty();
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        if (AuthHeaderHelper.hasValidHeader(AuthType.OAuth, headers)) {
            String token = AuthHeaderHelper.getAuthToken(AuthType.OAuth, headers);
            String userIp = GrpcUtils.realOrRemoteIp(attributes);
            return getOAuthToken(token, userIp);
        }
        return Optional.empty();
    }

    private Optional<AuthToken> getOAuthToken(String token, String userIp) {
        // TODO: get oauth scopes
        String[] scopes = {};
        return Optional.of(new OAuthToken(token, userIp, scopes));
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        try {
            OAuthToken oauthToken = (OAuthToken) token;
            return blackboxClient.oauth(oauthToken.getValue(), oauthToken.getScopes(), oauthToken.getUserIp())
                    .handle((userInfo, throwable) -> {
                        if (throwable == null) {
                            return new OAuthSubject(userInfo.getUid(), userInfo.getLogin());
                        }
                        var cause = CompletableFutures.unwrapCompletionException(throwable);
                        String message = "cannot authenticate by OAuth token, cause: " + cause.getMessage();
                        throw new AuthenticationException(message, cause);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }
}
