package ru.yandex.solomon.auth.openid;

import java.util.Base64;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;

import io.grpc.Attributes;
import io.grpc.Metadata;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.cloud.session.Session;
import ru.yandex.cloud.session.SessionClient;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.util.UrlBuilder;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * Must always be the last authenticator in a chain.
 *
 * @author Sergey Polovko
 */
public class OpenIdAuthenticator implements Authenticator {

    private final String domain;
    private final String clientId;
    private final String federationId;
    private final SessionClient sessionClient;

    public OpenIdAuthenticator(String domain, String clientId, String federationId, SessionClient sessionClient) {
        this.domain = domain;
        this.clientId = clientId;
        this.federationId = federationId;
        this.sessionClient = sessionClient;
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        String cookieHeader = Nullables.orEmpty(request.getHeaders().getFirst("Cookie"));
        return Optional.of(new OpenIdAuthToken(cookieHeader));
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        // grpc not supported
        return Optional.empty();
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        var openIdAuth = (OpenIdAuthToken) token;
        return sessionClient.check(domain, openIdAuth.getCookieHeader(), federationId)
                .thenApply(session -> {
                    if (session.isAbsent()) {
                        String state = generateState();
                        var authorizeUrl = new UrlBuilder(session.asAbsent().getRedirectUrl())
                                .appendQueryArg("response_type", "code")
                                .appendQueryArg("client_id", clientId)
                                .appendQueryArg("yc_federation_hint", federationId)
                                .appendQueryArg("scope", "openid")
                                .appendQueryArg("state", state)
                                .build();

                        throw new AuthenticationException("", authorizeUrl, state);
                    }

                    Session.Present p = session.asPresent();
                    return new OpenIdSubject(p.getAccountId(), p.getLogin());
                });
    }

    private static String generateState() {
        byte[] buf = new byte[20];
        ThreadLocalRandom.current().nextBytes(buf);
        return Base64.getUrlEncoder().encodeToString(buf);
    }
}
