package ru.yandex.solomon.auth.sessionid;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import io.grpc.Attributes;
import io.grpc.Metadata;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.NeedResetCookieException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.util.http.HttpUtils;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SessionIdAuthenticator implements Authenticator {
    private final BlackboxClient blackboxClient;

    public SessionIdAuthenticator(BlackboxClient blackboxClient) {
        this.blackboxClient = blackboxClient;
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        String userIp = HttpUtils.realOrRemoteIp(request);
        String sessionId = HttpUtils.cookieValue(request, "Session_id");
        if (sessionId == null) {
            return Optional.empty();
        }
        // sessionid2 is optional
        String sslSessionId = Strings.nullToEmpty(HttpUtils.cookieValue(request, "sessionid2"));
        return Optional.of(new SessionIdAuthToken(sessionId, sslSessionId, userIp));
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        // gRPC requests must not support session cookies
        return Optional.empty();
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        try {
            if (!(token instanceof SessionIdAuthToken)) {
                return CompletableFuture.failedFuture(new AuthenticationException("cannot authenticate by cookie, please check authentication type in the request"));
            }

            SessionIdAuthToken sessionIdAuthToken = (SessionIdAuthToken) token;

            String sessionId = sessionIdAuthToken.getSessionId();
            String sslSessionId = sessionIdAuthToken.getSslSessionId();
            String userIp = sessionIdAuthToken.getUserIp();

            return blackboxClient.sessionId(sessionId, sslSessionId, userIp)
                .handle((userInfo, throwable) -> {
                    if (throwable == null) {
                        return new SessionIdAuthSubject(userInfo.getLogin());
                    }
                    var cause = CompletableFutures.unwrapCompletionException(throwable);
                    String message = "cannot authenticate by SessionId cookie, cause: " + cause.getMessage();
                    if (cause instanceof NeedResetCookieException) {
                        throw new AuthenticationException(message, false, true);
                    }
                    throw new AuthenticationException(message, cause);
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }
}
