package ru.yandex.solomon.auth.tvm;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import io.grpc.Attributes;
import io.grpc.Metadata;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.grpc.utils.GrpcUtils;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;
import ru.yandex.passport.tvmauth.TicketStatus;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthToken;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.util.http.HttpUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.auth.http.AuthHeaderHelper.hasValidHeader;


/**
 * @author Sergey Polovko
 */
public class TvmAuthenticator implements Authenticator {

    private final TvmClient tvmClient;
    private final BlackboxClient blackboxClient;

    public TvmAuthenticator(TvmClient tvmClient, BlackboxClient blackboxClient) {
        this.tvmClient = tvmClient;
        this.blackboxClient = blackboxClient;
    }

    @Override
    public Optional<AuthToken> getToken(ServerHttpRequest request) {
        if (!hasValidHeader(AuthType.TvmService, request) && !hasValidHeader(AuthType.TvmUser, request)) {
            return Optional.empty();
        }

        String userTicket = request.getHeaders().getFirst(AuthType.TvmUser.getHeaderName());
        String serviceTicket = request.getHeaders().getFirst(AuthType.TvmService.getHeaderName());
        String userIp = HttpUtils.realOrRemoteIp(request);

        return getTvmToken(userTicket, serviceTicket, userIp);
    }

    @Override
    public Optional<AuthToken> getToken(Metadata headers, Attributes attributes) {
        if (!hasValidHeader(AuthType.TvmService, headers) && !hasValidHeader(AuthType.TvmUser, headers)) {
            return Optional.empty();
        }

        String userTicket = headers.get(AuthType.TvmUser.getMetadataKey());
        String serviceTicket = headers.get(AuthType.TvmService.getMetadataKey());
        String userIp = GrpcUtils.realOrRemoteIp(attributes);

        return getTvmToken(userTicket, serviceTicket, userIp);
    }

    private Optional<AuthToken> getTvmToken(@Nullable String userTicket, @Nullable String serviceTicket, String userIp) {
        if (StringUtils.isNotEmpty(userTicket)) {
            if (StringUtils.isEmpty(serviceTicket)) {
                throw new AuthenticationException(String.format(
                    "cannot authenticate by TVM, request must contain both %s and %s headers",
                    AuthType.TvmUser.getHeaderName(),
                    AuthType.TvmService.getHeaderName()));
            }

            return Optional.of(new TvmUserTicket(userTicket, serviceTicket, userIp));
        }

        if (StringUtils.isNotEmpty(serviceTicket)) {
            return Optional.of(new AuthToken(AuthType.TvmService, serviceTicket));
        }
        return Optional.empty();
    }

    @Override
    public CompletableFuture<AuthSubject> authenticate(AuthToken token) {
        try {
            if (token.getType() == AuthType.TvmUser) {
                checkServiceTicket(((TvmUserTicket) token).getServiceTicket());
                return checkUserTicket((TvmUserTicket) token);
            }

            if (token.getType() == AuthType.TvmService) {
                return completedFuture(checkServiceTicket(token.getValue()));
            }

            return failedFuture(new AuthenticationException(String.format(
                "cannot authenticate by TVM, request must contain at least %s header",
                AuthType.TvmService.getHeaderName())));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private AuthSubject checkServiceTicket(String ticketStr) {
        CheckedServiceTicket ticket = tvmClient.checkServiceTicket(ticketStr);
        try {
            if (ticket.getStatus() == TicketStatus.OK) {
                return TvmSubject.service(ticket.getSrc());
            }
            throw new AuthenticationException("cannot authenticate by TVM service ticket, status: " + ticket.getStatus());
        } finally {
        }
    }

    private CompletableFuture<AuthSubject> checkUserTicket(TvmUserTicket t) {
        CheckedUserTicket ticket = tvmClient.checkUserTicket(t.getUserTicket());
        try {
            if (ticket.getStatus() == TicketStatus.OK) {
                long uid = ticket.getDefaultUid();
                return blackboxClient.userInfo(uid, t.getUserIp())
                    .handle((userInfo, throwable) -> {
                        if (throwable == null) {
                            return TvmSubject.user(userInfo.getUid(), userInfo.getLogin());
                        }
                        var cause = CompletableFutures.unwrapCompletionException(throwable);
                        String message = "cannot authenticate by TVM user ticket, cause: " + cause.getMessage();
                        throw new AuthenticationException(message, cause);
                    });
            }
            String msg = "cannot authenticate by TVM user ticket, status: " + ticket.getStatus();
            return failedFuture(new AuthenticationException(msg));
        } finally {
        }
    }
}
